#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#define MAX_CHANNELS 8
#define CLIENTS 8
#define CALLERS 8
#define STATE_MS 1000
#define SETUP_MS 3000
#define HEARTBEAT_MS 10000
#define TIMEOUT_MS 60000

char default_client_bind[] = "127.0.0.1:1234";
char default_caller_bind[] = "127.0.0.1:1236";

char * client_bind, * caller_bind;
int client_sock, caller_sock;

// channels
unsigned int channel_cnt;
typedef struct s_channel {
  int caller_idx; // index of caller, -1 if none
  int client_idx; // index of client, -1 if none (must be -1 if caller_idx is -1)
  unsigned int ms_setup_sent;
} t_s_channel;
t_s_channel channels[MAX_CHANNELS];

// EBIP clients
typedef struct s_client {
  int used;
  struct sockaddr_in src_addr; // address packets are comng from
  struct sockaddr_in addr; // address to send packets to
  unsigned int ms_heartbeat_sent;
  unsigned int ms_state_sent;
  unsigned int ms_recvd;
  int channels[MAX_CHANNELS]; // if channel is in use by client
} t_s_client;
t_s_client clients[CLIENTS];

// callers
typedef struct s_caller {
  int used;
  int sock;
  int channel_idx; // index of channel, -1 if none yet
  char src[64]; // source phone number
  char dest[64]; // destination phone number
  char recv_buf[1024];
  unsigned int recv_buf_len;
  int line_too_long;
} t_s_caller;
t_s_caller callers[CALLERS];

// get time in milliseconds
static unsigned int get_ms( )
{
  struct timeval tv;
  gettimeofday( &tv, NULL );
  return (unsigned int)tv.tv_sec * 1000 + (unsigned int)tv.tv_usec / 1000;
}

// send string to socket
static void send_str( int sock, char * str )
{
  send( sock, str, strlen( str ), 0 );
}

// send message to a client
static void send_client( unsigned int client_idx, char * msg )
{
  sendto( client_sock, msg, strlen( msg ), 0, (struct sockaddr *)&clients[client_idx].addr, sizeof( clients[client_idx].addr ) );
  printf( "SEND (%d): %s\n", client_idx, msg );
}

// send message to all clients
static void send_clients( char * msg )
{
  unsigned int i;
  for( i = 0; i < CLIENTS; i++ )
    if( clients[i].used )
      sendto( client_sock, msg, strlen( msg ), 0, (struct sockaddr *)&clients[i].addr, sizeof( clients[i].addr ) );
  printf( "SEND (ALL): %s\n", msg );
}

// send heartbeats to all clients
static void send_heartbeats( )
{
  unsigned int i, time = get_ms( );
  for( i = 0; i < CLIENTS; i++ ) {
    if( clients[i].used ) {
      if( time - clients[i].ms_heartbeat_sent > HEARTBEAT_MS ) {
        send_client( i, "0:heartbeat" );
        clients[i].ms_heartbeat_sent = time;
      }
    }
  }
}

// send state to all clients
static void send_states( )
{
  unsigned int i, time = get_ms( );
  for( i = 0; i < CLIENTS; i++ ) {
    if( clients[i].used ) {
      if( time - clients[i].ms_state_sent > STATE_MS ) {
        int j;
        char msg[64];
        for( j = 0; j < (int)channel_cnt; j++ ) {
          snprintf( msg, sizeof( msg ), "%d:%s", j + 1, channels[j].caller_idx >= 0 ? "offhook" : "onhook" );
          send_client( i, msg );
        }
        clients[i].ms_state_sent = time;
      }
    }
  }
}

// send setup packets to clients
static void send_setups( )
{
  unsigned int i, time = get_ms( );
  for( i = 0; i < channel_cnt; i++ ) {
    if( channels[i].caller_idx >= 0 && channels[i].client_idx < 0 /* setup state */ ) {
      if( time - channels[i].ms_setup_sent > SETUP_MS ) {
        char msg[64];
        snprintf( msg, sizeof( msg ), "%d:setup:%s:%s", i + 1, callers[channels[i].caller_idx].src, callers[channels[i].caller_idx].dest );
        send_clients(  msg );
        channels[i].ms_setup_sent = time;
      }
    }
  }
}

// remove timed-out clients
static void remove_timed_out( )
{
  unsigned int i, time = get_ms( );
  for( i = 0; i < CLIENTS; i++ ) {
    if( clients[i].used ) {
      if( time - clients[i].ms_recvd > TIMEOUT_MS ) {
        printf( "client %d timed out\n", i );
        clients[i].used = 0;
      }
    }
  }
}

// client socket readable
static int client_sock_recv( )
{
  struct sockaddr_in recv_addr;
  socklen_t recv_addr_len;
  int len, i;
  unsigned int client_idx;
  int chan;
  char buffer[1024], command[64];

  // receive from socket
  recv_addr_len = sizeof( recv_addr );
  len = recvfrom( client_sock, &buffer, sizeof( buffer ) - 1, 0, (struct sockaddr *)&recv_addr, &recv_addr_len );
  if( len < 0 ) {
    perror( "recvfrom (client)" );
    return -1;
  }
  // replace unprintable characters
  for( i = 0; i < len; i++ )
    if( buffer[i] < ' ' || buffer[i] > '~' )
      buffer[i] = '_';
  // terminate string
  buffer[len] = 0;
  // find client
  for( client_idx = 0; client_idx < CLIENTS; client_idx++ )
    if( clients[client_idx].used && memcmp( &clients[client_idx].src_addr, &recv_addr, sizeof( recv_addr ) ) == 0 )
      break;
  // output string received
  if( client_idx < CLIENTS )
    printf( "RECV (%d): %s\n", client_idx, buffer );
  else
    printf( "RECV (NEW): %s\n", buffer );
  // parse channel and command
  if( sscanf( buffer, "%u:%60[^:]", &chan, command ) == 2 )
  {
    chan--;
    // register command of unknown client and still free client slots
    if( strcmp( command, "register" ) == 0 && client_idx >= CLIENTS )
    {
      // find free client slot
      for( client_idx = 0; client_idx < CLIENTS; client_idx++ )
        if( ! clients[client_idx].used )
          break;
      if( client_idx < CLIENTS ) {
        // add client
        clients[client_idx].used = 1;
        for( i = 0; i < (int)channel_cnt; i++ )
          clients[client_idx].channels[i] = 0;
        memcpy( &clients[client_idx].src_addr, &recv_addr, sizeof( recv_addr ) );
        memcpy( &clients[client_idx].addr, &recv_addr, sizeof( recv_addr ) );
        printf( "new client %d\n", client_idx );
      }
    }
    // client available
    if( client_idx < CLIENTS )
    {
      // remember reception
      clients[client_idx].ms_recvd = get_ms( );
      // commands
      if( strcmp( command, "register" ) == 0 ) {
        unsigned int port = 0;
        // parse port
        if( sscanf( buffer, "%*u:%*60[^:]:%u", &port ) == 1 && port > 0 && port < 65536 )
          clients[client_idx].addr.sin_port = htons( port );
        // send answer to client
        send_client( client_idx, "0:heartbeat" );
        clients[client_idx].ms_heartbeat_sent = get_ms( );
        {
          char msg[64];
          int i;
          for( i = 0; i < (int)channel_cnt; i++ ) {
            snprintf( msg, sizeof( msg ), "%d:%s", i + 1, channels[i].caller_idx >= 0 ? "offhook" : "onhook" );
            send_client( client_idx, msg );
            clients[client_idx].ms_state_sent = get_ms( );
          }
        }
      // accept
      } else if( strcmp( command, "accept" ) == 0 ) {
        // check that channel is unaccepted yet
        if( chan >= 0 && chan < (int)channel_cnt && channels[chan].client_idx == -1 ) {
          // assign channel to client
          channels[chan].client_idx = client_idx;
          clients[client_idx].channels[chan] = 1;
          // send answer to client
          char msg[64];
          snprintf( msg, sizeof( msg ), "%u:connected", chan + 1 );
          send_client( client_idx, msg );
          // send connected message to caller
          int caller_idx = channels[chan].caller_idx;
          send_str( callers[caller_idx].sock, "connected\n" );
          // send state to client
          snprintf( msg, sizeof( msg ), "%u:offhook", chan + 1 );
          send_client( client_idx, msg );
          clients[client_idx].ms_state_sent = get_ms( );
        }
      // playback(ground)
      } else if( strcmp( command, "playback" ) == 0 || strcmp( command, "playbackground" ) == 0 ) {
        // parse sound name
        char sound[256];
        if( sscanf( buffer, "%*u:%*60[^:]:%250[^:]", sound ) != 1 )
          sound[0] = 0;
        // check that channel is connected
        if( chan >= 0 && chan < (int)channel_cnt && channels[chan].client_idx >= 0 ) {
          // send sound message to caller
          int caller_idx = channels[chan].caller_idx;
          char msg[384];
          snprintf( msg, sizeof( msg ), "playing sound: %s\n", sound );
          send_str( callers[caller_idx].sock, msg );
        }
      // hangup
      } else if( strcmp( command, "hangup" ) == 0 ) {
        // parse reason name
        char reason[256];
        if( sscanf( buffer, "%*u:%*60[^:]:%250[^:]", reason ) != 1 )
          reason[0] = 0;
        // check that channel is connected
        if( chan >= 0 && chan < (int)channel_cnt && channels[chan].client_idx >= 0 ) {
          // send disconnected message to caller
          int caller_idx = channels[chan].caller_idx;
          char msg[384];
          snprintf( msg, sizeof( msg ), "disconnected: %s\n", reason );
          send_str( callers[caller_idx].sock, msg );
          // close channel
          clients[client_idx].channels[chan] = 0;
          channels[chan].client_idx = -1;
          channels[chan].caller_idx = -1;
          // remove caller
          printf( "caller %d disconnected\n", caller_idx );
          close( callers[caller_idx].sock );
          callers[caller_idx].used = 0;
          // send state to client
          snprintf( msg, sizeof( msg ), "%u:onhook", chan + 1 );
          send_client( client_idx, msg );
          clients[client_idx].ms_state_sent = get_ms( );
        }
      }
    } // if( client_idx ...
  } // if( sscanf( ...

  return 0;
}

// caller socket readable (i.e. pending connection)
static int caller_sock_accept( )
{
  int sock;
  struct sockaddr_in addr;
  socklen_t addr_len;

  // accept connection
  addr_len = sizeof( addr );
  sock = accept( caller_sock, (struct sockaddr*)&addr, &addr_len );
  send_str( sock, "phonesim2\n" );
  if( sock != -1 ) {
    unsigned int caller_idx;

    // find new caller slot
    for( caller_idx = 0; caller_idx < CALLERS; caller_idx++ )
      if( ! callers[caller_idx].used )
        break;
    if( caller_idx >= CALLERS ) {
      send_str( sock, "too many callers\n" );
      close( sock );
    } else {

      // add caller
      printf( "new caller %d\n", caller_idx );
      callers[caller_idx].used = 1;
      callers[caller_idx].sock = sock;
      callers[caller_idx].channel_idx = -1;
      callers[caller_idx].src[0] = 0;
      callers[caller_idx].dest[0] = 0;
      callers[caller_idx].recv_buf_len = 0;
      callers[caller_idx].line_too_long = 0;
      send_str( sock, "enter your own phone number:\n" );

    }
  }

  return 0;
}

static int caller_sock_recv( int caller_idx )
{
  t_s_caller * caller;
  int len;

  caller = &callers[caller_idx];

  // receive from socket
  len = recv( caller->sock,
              caller->recv_buf + caller->recv_buf_len,
              sizeof( caller->recv_buf ) - caller->recv_buf_len,
              0 );
  if( len < 0 ) {
    perror( "recv (caller)" );
    return -1;
  }
  caller->recv_buf_len += len;

  // end of connection
  if( caller->recv_buf_len == 0 ) {
    // close channel
    if( caller->channel_idx >= 0 ) {
      channels[caller->channel_idx].caller_idx = -1;
      if( channels[caller->channel_idx].client_idx >= 0 ) {
        int client_idx = channels[caller->channel_idx].client_idx;
        channels[caller->channel_idx].client_idx = -1;
        // close channel
        int chan = caller->channel_idx;
        clients[client_idx].channels[chan] = 0;
        channels[chan].client_idx = -1;
        channels[chan].caller_idx = -1;
        // send state to client
        char msg[64];
        snprintf( msg, sizeof( msg ), "%u:onhook", chan + 1 );
        send_client( client_idx, msg );
        clients[client_idx].ms_state_sent = get_ms( );
      }
    }
    // remove caller
    printf( "caller %d quit\n", caller_idx );
    close( caller->sock );
    caller->used = 0;
    return 0;
  }

  // process lines in receive buffer
  for( ; ; ) {
    char * end, * line, * next;
    unsigned int line_len;
    int i;

    // find end of line and get line
    end = memchr( caller->recv_buf, '\r', caller->recv_buf_len );
    if( end != NULL )
      *end = 0;
    end = memchr( caller->recv_buf, '\n', caller->recv_buf_len );
    if( end != NULL )
      *end = 0;
    end = memchr( caller->recv_buf, '\0', caller->recv_buf_len );
    if( end == NULL )
      break;
    line = caller->recv_buf;
    line_len = end - caller->recv_buf + 1;
    next = end + 1;

    // ignore lines too long
    if( caller->line_too_long )
      caller->line_too_long = 0;
    // ignore empty lines
    else if( line[0] != 0 ) {

      // process line - source number
      if( caller->src[0] == 0 ) {
        // store number
        if( strlen( line ) > sizeof( caller->src ) - 1 )
          line[sizeof( caller->src ) - 1] = 0;
        strcpy( caller->src, line );
        // ask for destination number
        send_str( caller->sock, "enter the phone number to call:\n" );

      // process line - destination number
      } else if( caller->dest[0] == 0 ) {
        // store number
        if( strlen( line ) > sizeof( caller->dest ) - 1 )
          line[sizeof( caller->dest ) - 1] = 0;
        strcpy( caller->dest, line );
        // find free channel
        for( i = 0; i < (int)channel_cnt; i++ )
          if( channels[i].caller_idx < 0 )
            break;
        if( i >= (int)channel_cnt ) {
          send_str( caller->sock, "busy: no more free channels\n" );
          printf( "caller %d rejected\n", caller_idx );
          close( caller->sock );
          caller->used = 0;
          return 0;
        } else {
          // use this channel
          send_str( caller->sock, "please wait while calling\n" );
          caller->channel_idx = i;
          channels[i].caller_idx = caller_idx;
          channels[i].client_idx = -1;
          channels[i].ms_setup_sent = get_ms( ) - SETUP_MS;
        }

      // process line - key
      } else {
        // check if connected yet
        int chan = caller->channel_idx;
        if( channels[chan].client_idx >= 0 ) {
          // parse keys
	  char * ptr;
	  int sent = 0, ignored = 0;
	  for( ptr = line; *ptr != 0; ptr++ ) {
            char key = *ptr;
            if( strchr( "0123456789*#", key ) != NULL ) {
              // send dtmf key to client
              int client_idx = channels[chan].client_idx;
              char msg[64];
              snprintf( msg, sizeof( msg ), "%u:dtmf:%c", chan + 1, key );
              send_client( client_idx, msg );
              // accept key
              snprintf( msg, sizeof( msg ), "dtmf key \"%c\" sent\n", key );
              send_str( caller->sock, msg );
              sent++;
            } else
              ignored++;
          }
          // report error if invalid dtmf keys were detected
          if( ignored > 0 ) {
            if( sent > 0 )
              send_str( caller->sock, "invalid dtmf keys on line were ignored\n" );
            else
              send_str( caller->sock, "line contained only invalid dtmf keys\n" );
          }
        } else {
          // refuse sending key if not connected yet
          send_str( caller->sock, "not connected to client yet\n" );
        }
      }

    } // ignore empty lines

    // remove line from receive buffer
    caller->recv_buf_len -= line_len;
    memmove( caller->recv_buf, caller->recv_buf + line_len, caller->recv_buf_len );

  } // process lines in receive buffer

  // receive buffer full
  if( callers[caller_idx].recv_buf_len >= sizeof( callers[caller_idx].recv_buf ) ) {
    callers[caller_idx].recv_buf_len = 0;
    callers[caller_idx].line_too_long = 1;
  }

  return 0;
}

// main
int main( int argc, char * * argv )
{
  int i;
  struct sockaddr_in client_bind_addr, caller_bind_addr;
  char * ip, * port;

  // get parameters
  
  printf( "phonesim2 - quickhack by 1stein <1stein@schuermans.info>\n" );
  if( argc < 4 )
    printf( "call with <#channels> <client-bind-address> <caller-bind-address>\n" );
  
  if( argc < 2 )
    channel_cnt = 2;
  else
    channel_cnt = atoi( argv[1] );
  if( channel_cnt > MAX_CHANNELS )
    channel_cnt = MAX_CHANNELS;
  if( argc < 3 )
    client_bind = default_client_bind;
  else
    client_bind = argv[2];
  if( argc < 4 )
    caller_bind = default_caller_bind;
  else
    caller_bind = argv[3];

  printf( "channels=%d client-bind-address=%s caller-bind-address=%s\n", channel_cnt, client_bind, caller_bind );

  // initialize channels
  for( i = 0; i < (int)channel_cnt; i++ )
    channels[i].caller_idx = -1;

  // get bind addresses
  ip = client_bind;
  port = strchr( ip, ':' );
  if( port == NULL ) {
    port = ip;
    ip = "0.0.0.0";
  } else {
    *port = 0;
    port++;
  }
  client_bind_addr.sin_family = AF_INET;
  client_bind_addr.sin_port = htons( atoi( port ) );
  if( inet_aton( ip, &client_bind_addr.sin_addr ) == 0 )
    client_bind_addr.sin_addr.s_addr = htonl( INADDR_ANY );
  ip = caller_bind;
  port = strchr( ip, ':' );
  if( port == NULL ) {
    port = ip;
    ip = "0.0.0.0";
  } else {
    *port = 0;
    port++;
  }
  caller_bind_addr.sin_family = AF_INET;
  caller_bind_addr.sin_port = htons( atoi( port ) );
  if( inet_aton( ip, &caller_bind_addr.sin_addr ) == 0 )
    caller_bind_addr.sin_addr.s_addr = htonl( INADDR_ANY );

  // create sockets and bind / listen
  client_sock = socket( AF_INET, SOCK_DGRAM, IPPROTO_UDP );
  if( caller_sock == -1 ) {
    perror( "could not create client socket" );
    return -1;
  }
  if( bind( client_sock, (struct sockaddr *)&client_bind_addr, sizeof( client_bind_addr ) ) == -1 ) {
    perror( "could not bind client socket" );
    close( client_sock );
    return -1;
  }
  caller_sock = socket( AF_INET, SOCK_STREAM, IPPROTO_TCP );
  if( caller_sock == -1 ) {
    perror( "could not create caller socket" );
    close( client_sock );
    return -1;
  }
  if( bind( caller_sock, (struct sockaddr *)&caller_bind_addr, sizeof( caller_bind_addr ) ) == -1 ) {
    perror( "could not bind caller socket" );
    close( caller_sock );
    close( client_sock );
    return -1;
  }
  if( listen( caller_sock, SOMAXCONN ) == -1 ) {
    perror( "could not listen on caller socket" );
    close( caller_sock );
    close( client_sock );
    return -1;
  }

  for( ; ; )
  {
    int max_fd, i;
    fd_set read_fds;
    struct timeval timeout;

    // check if it is possible to read from sockets
    max_fd = 0;
    FD_ZERO( &read_fds );
    if( client_sock > max_fd )
      max_fd = client_sock;
    FD_SET( client_sock, &read_fds );
    if( caller_sock > max_fd )
      max_fd = caller_sock;
    FD_SET( caller_sock, &read_fds );
    for( i = 0; i < CALLERS; i++ ) {
      if( callers[i].used ) {
        if( callers[i].sock > max_fd )
          max_fd = callers[i].sock;
        FD_SET( callers[i].sock, &read_fds );
      }
    }
    timeout.tv_usec = 20000; // 20ms
    timeout.tv_sec = 0;
    i = select( max_fd + 1, &read_fds, NULL, NULL, &timeout );
    if( i == -1 ) {
      if( errno != EINTR ) { // ignore the error "interrupted from system-call"
        perror( "select" );
        break;
      }
      FD_ZERO( &read_fds ); // interrupted, so empty fd-sets
    }

    // client socket readable
    if( FD_ISSET( client_sock, &read_fds ) )
      if( client_sock_recv( ) == -1 )
        break;

    // caller socket readable (i.e. pending connection)
    if( FD_ISSET( caller_sock, &read_fds ) )
      if( caller_sock_accept( ) == -1 )
        break;
    
    // caller connection socket readable
    for( i = 0; i < CALLERS; i++ )
      if( callers[i].used )
        if( FD_ISSET( callers[i].sock, &read_fds ) )
          if( caller_sock_recv( i ) == -1 )
            break;
    if( i < CALLERS )
      break;

    // do periodic stuff
    send_heartbeats( );
    send_states( );
    send_setups( );
    remove_timed_out( );

  } // for( ; ; )

  // close all sockets
  for( i = 0; i < CALLERS; i++ ) {
    if( callers[i].used ) {
      send_str( callers[i].sock, "exiting\n" );
      close( callers[i].sock );
    }
  }
  close( caller_sock );
  close( client_sock );

  return 0;
}

