/* blib - Library of useful things to hack the Blinkenlights
 *
 * Copyright (c) 2001-2002  Sven Neumann <sven@gimp.org>
 *                          Michael Natterer <mitch@gimp.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <string.h>

#include <glib-object.h>

#include <blib/blib.h>

#include "digits.h"


#define PONG_START_TIMEOUT      200
#define COMPUTER_LOOSE_SPEEDUP   10


typedef enum
{
  RIGHT,
  LEFT
} XDirection;

typedef enum
{
  UP,
  DOWN
} YDirection;


typedef enum
{
  GAME_INIT,
  TPLAYER_JOIN,
  BPLAYER_JOIN,
  GAME_OVER
} AnimType;


#define B_TYPE_PONG            (b_type_pong)
#define B_PONG_V(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), B_TYPE_PONG, BPongV))
#define B_PONG_V_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass), B_TYPE_PONG, BPongVClass))
#define B_IS_PONG(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), B_TYPE_PONG))
#define B_IS_PONG_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), B_TYPE_PONG))

typedef struct _BPongV      BPongV;
typedef struct _BPongVClass BPongVClass;

struct _BPongV
{
  BModule       parent_instance;

  AnimType      anim;
  gint          anim_steps;

  gint          paddle_size;
  gint          paddle_step;

  gint          tpaddle;
  gint          bpaddle;
  gint          ball_x;
  gint          ball_y;
  XDirection    ball_xdir;
  YDirection    ball_ydir;

  gint          tplayer_score;
  gint          bplayer_score;

  gint          tplayer_device_id;
  gint          bplayer_device_id;

  gint          timeout;
};

struct _BPongVClass
{
  BModuleClass  parent_class;
};


static GType      b_pong_v_get_type      (GTypeModule   *module);

static void       b_pong_v_class_init    (BPongVClass    *klass);
static void       b_pong_v_init          (BPongV         *pong);

static gboolean   b_pong_v_query         (gint           width,
                                        gint           height,
                                        gint           channels,
                                        gint           maxval);
static gboolean   b_pong_v_prepare       (BModule       *module,
                                        GError       **error);
static void       b_pong_v_relax         (BModule       *module);
static void       b_pong_v_start         (BModule       *module);
static void       b_pong_v_stop          (BModule       *module);
static void       b_pong_v_event         (BModule       *module,
                                        BModuleEvent  *event);
static gint       b_pong_v_tick          (BModule       *module);
static void       b_pong_v_describe      (BModule       *module,
                                        const gchar  **title,
                                        const gchar  **description,
                                        const gchar  **author);

static void       b_pong_v_init_game     (BPongV         *pong);
static gint       b_pong_v_move_ball     (BPongV         *pong);
static void       b_pong_v_computer_move (BPongV         *pong,
                                        gint          *paddle);
static gboolean   b_pong_v_reflect       (BPongV         *pong,
                                        gint           paddle);
static void       b_pong_v_draw          (BPongV         *pong,
                                        gint           lpaddle,
                                        gint           rpaddle,
                                        gint           ball_x,
                                        gint           ball_y);
static void       b_pong_v_draw_scores   (BPongV         *pong);


static GType b_type_pong = 0;


G_MODULE_EXPORT gboolean
b_module_register (GTypeModule *module)
{
  b_pong_v_get_type (module);
  return TRUE;
}

static GType
b_pong_v_get_type (GTypeModule *module)
{
  if (! b_type_pong)
    {
      static const GTypeInfo pong_info =
      {
        sizeof (BPongVClass),
        NULL,           /* base_init */
        NULL,           /* base_finalize */
        (GClassInitFunc) b_pong_v_class_init,
        NULL,           /* class_finalize */
        NULL,           /* class_data */
        sizeof (BPongV),
        0,              /* n_preallocs */
        (GInstanceInitFunc) b_pong_v_init,
      };

      b_type_pong = g_type_module_register_type (module,
                                                 B_TYPE_MODULE, "BPongV",
                                                 &pong_info, 0);
    }

  return b_type_pong;
}

static void
b_pong_v_class_init (BPongVClass *klass)
{
  BModuleClass *module_class;

  module_class = B_MODULE_CLASS (klass);

  module_class->max_players = 2;

  module_class->query    = b_pong_v_query;
  module_class->prepare  = b_pong_v_prepare;
  module_class->relax    = b_pong_v_relax;
  module_class->start    = b_pong_v_start;
  module_class->stop     = b_pong_v_stop;
  module_class->event    = b_pong_v_event;
  module_class->tick     = b_pong_v_tick;
  module_class->describe = b_pong_v_describe;
}

static void
b_pong_v_init (BPongV *pong)
{
  pong->anim_steps        = 0;

  pong->tplayer_device_id = -1;
  pong->bplayer_device_id = -1;
}

static gboolean
b_pong_v_query (gint     width,
              gint     height,
              gint     channels,
              gint     maxval)
{
  return (width > 7 && height > 7 && channels == 1);
}

static gboolean
b_pong_v_prepare (BModule  *module,
                GError  **error)
{
  BPongV *pong = B_PONG_V (module);

  pong->paddle_size = MAX (3, module->width / 4);
  pong->paddle_step = MAX (1, pong->paddle_size / 3);

  return TRUE;
}

static void
b_pong_v_relax (BModule *module)
{
}

static void
b_pong_v_start (BModule *module)
{
  BPongV *pong = B_PONG_V (module);

  pong->tpaddle       = pong->bpaddle       = module->width / 2 - 1;
  pong->tplayer_score = pong->bplayer_score = 0;

  b_pong_v_init_game (pong);

  pong->timeout = PONG_START_TIMEOUT;

  b_module_ticker_start (module, pong->timeout);
}

static void
b_pong_v_stop (BModule *module)
{
  BPongV *pong = B_PONG_V (module);

  pong->tplayer_device_id = -1;
  pong->bplayer_device_id = -1;
}

static void
b_pong_v_event (BModule      *module,
              BModuleEvent *event)
{
  BPongV *pong = B_PONG_V (module);

  switch (event->type)
    {
    case B_EVENT_TYPE_KEY:
      if (pong->anim_steps)
        return;

      switch (event->key)
        {
        case B_KEY_1:
        case B_KEY_2:
        case B_KEY_4:
        case B_KEY_5:
        case B_KEY_7:
        case B_KEY_8:
          /* left */
          if (event->device_id == pong->tplayer_device_id)
            {
              pong->tpaddle = MAX (pong->tpaddle - pong->paddle_step, 0);
            }
          else if (event->device_id == pong->bplayer_device_id)
            {
              pong->bpaddle = MAX (pong->bpaddle - pong->paddle_step, 0);
            }
          break;

        case B_KEY_3:
        case B_KEY_6:
        case B_KEY_9:
          /* right */
          if (event->device_id == pong->tplayer_device_id)
            {
              pong->tpaddle = MIN (pong->tpaddle + pong->paddle_step,
                                   module->width - pong->paddle_size);
            }
          else if (event->device_id == pong->bplayer_device_id)
            {
              pong->bpaddle = MIN (pong->bpaddle + pong->paddle_step,
                                   module->width - pong->paddle_size);
            }
          break;

        default:
          break;
        }
      break;

    case B_EVENT_TYPE_PLAYER_ENTERED:
      if (pong->tplayer_device_id == -1)
        {
          pong->anim              = TPLAYER_JOIN;
          pong->anim_steps        = 6;
          pong->tplayer_device_id = event->device_id;

          module->num_players++;
        }
      else if (pong->bplayer_device_id == -1)
        {
          pong->anim              = BPLAYER_JOIN;
          pong->anim_steps        = 6;
          pong->bplayer_device_id = event->device_id;

          module->num_players++;
        }
      break;

    case B_EVENT_TYPE_PLAYER_LEFT:
      if (pong->tplayer_device_id == event->device_id)
        {
          pong->tplayer_device_id = -1;

          module->num_players--;
        }
      else if (pong->bplayer_device_id == event->device_id)
        {
          pong->bplayer_device_id = -1;

          module->num_players--;
        }
      break;

    default:
      break;
    }
}

static gint
b_pong_v_tick (BModule *module)
{
  BPongV *pong = B_PONG_V (module);

  if (pong->anim_steps > 0)
    {
      pong->anim_steps--;

      switch (pong->anim)
        {
        case GAME_INIT:
          b_pong_v_draw (pong,
                       pong->tpaddle, pong->bpaddle,
                       pong->anim_steps == 0 ? pong->ball_x : -1,
                       pong->anim_steps == 0 ? pong->ball_y : -1);
          break;
        case TPLAYER_JOIN:
          b_pong_v_draw (pong,
                       (pong->anim_steps & 1) ? pong->tpaddle : -1,
                       pong->bpaddle,
                       -1, -1);
          break;
        case BPLAYER_JOIN:
          b_pong_v_draw (pong,
                       pong->tpaddle,
                       (pong->anim_steps & 1) ? pong->bpaddle : -1,
                       -1, -1);
          break;
        case GAME_OVER:
          b_pong_v_draw (pong,
                       pong->tpaddle, pong->bpaddle,
                       (pong->anim_steps & 1) ? pong->ball_x : -1,
                       (pong->anim_steps & 1) ? pong->ball_y : -1);

          if (pong->anim_steps == 0)
            {
              if ((pong->tplayer_score != pong->bplayer_score) &&
                  (pong->tplayer_score >= 10 || pong->bplayer_score >= 10))
                {
                  b_module_request_stop (module);
                  return 0;
                }
              else
                b_pong_v_init_game (pong);
            }

          break;
        }
    }
  else
    {
      switch (b_pong_v_move_ball (pong))
        {
        case -1:
          pong->tplayer_score++;
          pong->anim_steps = 6;
          pong->anim = GAME_OVER;
	  if (pong->tplayer_device_id == -1) /* computer lost */
	    pong->timeout -= COMPUTER_LOOSE_SPEEDUP;
          break;

        case 1:
          pong->bplayer_score++;
          pong->anim_steps = 6;
          pong->anim = GAME_OVER;
	  if (pong->bplayer_device_id == -1) /* computer lost */
	    pong->timeout -= COMPUTER_LOOSE_SPEEDUP;
          break;

        case 0:
          if (pong->tplayer_device_id == -1)
            b_pong_v_computer_move (pong, &pong->tpaddle);
          if (pong->bplayer_device_id == -1)
            b_pong_v_computer_move (pong, &pong->bpaddle);
          break;
        }

      b_pong_v_draw (pong,
                   pong->tpaddle, pong->bpaddle,
                   pong->ball_x, pong->ball_y);
    }

  return pong->timeout;
}

static void
b_pong_v_describe (BModule      *module,
                 const gchar **title,
                 const gchar **description,
                 const gchar **author)
{
  *title       = "BPongV";
  *description = "vertical Pong game";
  *author      = "Sven Neumann";
}

static void
b_pong_v_init_game (BPongV *pong)
{
  BModule *module;
  gint     paddle;
  gint     foo;

  module = B_MODULE (pong);

  foo = rand ();

  if (foo & 0x1)
    {
      paddle = pong->tpaddle;
      pong->ball_y    = 0;
      pong->ball_ydir = DOWN;
    }
  else
    {
      paddle = pong->bpaddle;
      pong->ball_y    = module->height - 1;
      pong->ball_ydir = UP;
    }

  if ((foo & 0x2 && paddle != 0) ||
      paddle == module->width - pong->paddle_size)
    {
      pong->ball_x    = 0;
      pong->ball_xdir = RIGHT;
    }
  else
    {
      pong->ball_x    = module->width - 1;
      pong->ball_xdir = LEFT;
    }

  pong->anim       = GAME_INIT;
  pong->anim_steps = 4;
}

static gint
b_pong_v_move_ball (BPongV *pong)
{
  BModule *module = B_MODULE (pong);

  switch (pong->ball_ydir)
    {
    case DOWN:
      pong->ball_y++;
      break;
    case UP:
      pong->ball_y--;
      break;
    }
  switch (pong->ball_xdir)
    {
    case LEFT:
      pong->ball_x--;
      break;
    case RIGHT:
      pong->ball_x++;
      break;
    }

  /* collision with walls ? */
  if (pong->ball_x < 0)
    {
      pong->ball_x = 1;
      pong->ball_xdir = RIGHT;
    }
  else if (pong->ball_x >= module->width)
    {
      pong->ball_x = module->width - 2;
      pong->ball_xdir = LEFT;
    }

  /* collision with top paddle or out ? */
  if (pong->ball_y <= 0)
    {
      if (! b_pong_v_reflect (pong, pong->tpaddle))
        return 1;  /* bottom wins */

      pong->ball_y = 2;
      pong->ball_ydir = DOWN;
    }
  /* collision with bottom paddle or out ? */
  else if (pong->ball_y >= module->height - 1)
    {
      if (! b_pong_v_reflect (pong, pong->bpaddle))
        return -1;  /* left wins */
      pong->ball_y = module->height - 2;
      pong->ball_ydir = UP;
    }

  return 0;
}

static void
b_pong_v_computer_move (BPongV *pong,
                      gint  *paddle)
{
  BModule *module = B_MODULE (pong);

  if (rand () & 1)
    return;

  if (*paddle - pong->ball_x > -1)
    (*paddle)--;
  else if (*paddle - pong->ball_x < 1)
    (*paddle)++;

  *paddle = CLAMP (*paddle, 0, module->width - pong->paddle_size);
}

static gboolean
b_pong_v_reflect (BPongV *pong,
                gint   paddle)
{
  gint d = pong->ball_x - paddle;

  switch (pong->ball_xdir)
    {
    case RIGHT: /* we hit the paddle coming from the left */

      if (d < 0)
        return FALSE;
      else if (d == 0)
        {
          pong->ball_xdir = LEFT;
          pong->ball_x -= 2;
          return TRUE;
        }
      else if (d <= pong->paddle_size)
        return TRUE;
      else
        return FALSE;

      break;

    case LEFT: /* we hit the paddle coming from the right */

      if (d < -1)
        return FALSE;
      else if (d < pong->paddle_size - 1)
        return TRUE;
      else if (d == pong->paddle_size - 1)
        {
          pong->ball_xdir = RIGHT;
          pong->ball_x += 2;
          return TRUE;
        }
      else
        return FALSE;

      break;
    }

  return TRUE;
}

static void
b_pong_v_draw (BPongV *pong,
             gint   tpaddle,
             gint   bpaddle,
             gint   ball_x,
             gint   ball_y)
{
  BModule *module;
  gint     width;
  gint     height;
  gint     i;

  module = B_MODULE (pong);

  width  = module->width;
  height = module->height;

  b_module_fill (module, 0);

  if (tpaddle >= 0 && tpaddle <= width - pong->paddle_size)
    for (i = 0; i < pong->paddle_size; i++)
      b_module_draw_point (module, tpaddle + i, 0, module->maxval);

  if (bpaddle >= 0 && bpaddle <= width - pong->paddle_size)
    for (i = 0; i < pong->paddle_size; i++)
      b_module_draw_point (module, bpaddle + i, height - 1, module->maxval);

  if (ball_x >= 0 && ball_x < width && ball_y >= 0 && ball_y < height)
    b_module_draw_point (module, ball_x, ball_y, module->maxval);

  if (pong->anim == GAME_OVER)
    b_pong_v_draw_scores (pong);

  b_module_paint (module);
}

static void
b_pong_v_draw_scores (BPongV *pong)
{
  BModule     *module;
  gchar       *ttext, *btext;
  gint         tlen, blen;
  gint         x0, y0;
  gint         x, y;
  gint         n, i;
  const BFont *digits = &b_digits_3x5;
   
 
  module = B_MODULE (pong);

  ttext = g_strdup_printf ("%d", pong->tplayer_score);
  btext = g_strdup_printf ("%d", pong->bplayer_score);
  tlen = strlen (ttext);
  blen = strlen (btext);

  x0 = (module->width - tlen * digits->advance) / 2;
  y0 = (module->height / 3) - 4;
  for (n = 0; n < tlen; n++)
    {
      for (i = 0; i < digits->num_digits && digits->digits_str[i] != ttext[n]; i++);

      if (i < digits->num_digits)
        for (x = 0; x < digits->width; x++)
          for (y = 0; y < digits->height; y++)
            if (digits->data[i][y * digits->width + x] != '0')
              b_module_draw_point (module, x0 + x, y0 + y, module->maxval);

      x0 += digits->advance;
    }

  x0 = (module->width - blen * digits->advance) / 2;
  y0 = (module->height - module->height / 3) - 4;
  for (n = 0; n < blen; n++)
    {
      for (i = 0; i < digits->num_digits && digits->digits_str[i] != btext[n]; i++);

      if (i < digits->num_digits)
        for (x = 0; x < digits->width; x++)
          for (y = 0; y < digits->height; y++)
            if (digits->data[i][y * digits->width + x] != '0')
              b_module_draw_point (module, x0 + x, y0 + y, module->maxval);

      x0 += digits->advance;
    }

  g_free (ttext);
  g_free (btext);
}
