/* BLINKENLIGHTS PROXY
 * version 0.9 date 2003-12-23
 * Copyright (C) 2003 1stein <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <malloc.h>
#include <sys/types.h>
#include <sys/socket.h>

#include "addr.h"
#include "global.h"
#include "list.h"
#include "olist.h"
#include "outport.h"
#include "output.h"
#include "proto.h"
#include "stream.h"
#include "infilter.h"

//type for data of input filter list items (see header file)
struct s_infilter_data
{
  st_addr_range range; //the address range to match
  et_proto proto; //the protocol to match
  st_format format; //the format to match
  char name[list_item_name_size]; //name for the stream
  char stop; //!=0: stop the input filtering after this entry
  char op_name[list_item_name_size]; //name of the output port to pass matching streams to
  unsigned short priority; //priority
  unsigned short timeout; //timeout (in seconds)
};

//type for a input filter list (see header file)
struct s_infilter_list
{
  st_olist * p_olist; //the input filter list
  int af; //the address family of this list (obtained from owning port)
};

//create a new input filter list data item
//returns the pointer to the data item or NULL in case of error
static st_infilter_data * infilter_data_new( )
{
  st_infilter_data * p_data;

  //allocate a new input filter list data item
  p_data = (st_infilter_data *)malloc( sizeof( st_infilter_data ) );
  if( p_data == NULL )
    return NULL;

  //initialize data item
  p_data->range.af = AF_UNSPEC; //no address range yet
  p_data->proto = proto_none; //no protocol yet
  p_data->format = format_none; //no format yet
  p_data->name[0] = 0; //no name yet
  p_data->stop = 0; //do not stop input filtering after this entry
  p_data->op_name[0] = 0; //no ooutput port to pass matching streams to
  p_data->priority = 0; //lowest priority
  p_data->timeout = 0; //immediate timeout

  return p_data;
}

//free input filter list data item
static void infilter_data_free( void * vp_data )
{
  st_infilter_data * p_data = vp_data;

  //free data item
  free( p_data );
}

//create a new input filter list
//<af> specifies the address family of this input filter list (obtained from owning port)
//returns the pointer to the list or NULL in case of error
st_infilter_list * infilter_new( int af, st_out_info * p_out_info )
{
  st_infilter_list * p_infilter_list;

  //create structure
  p_infilter_list = (st_infilter_list *)malloc( sizeof( st_infilter_list ) );
  if( p_infilter_list == NULL )
  {
    out_str( p_out_info, "  error: could not create new input filter list (out of memory)\n" );
    return NULL;
  }

  //create list
  p_infilter_list->p_olist = olist_new( );
  if( p_infilter_list->p_olist == NULL )
  {
    free( p_infilter_list );
    out_str( p_out_info, "  error: could not create new input filter list\n" );
    return NULL;
  }

  //initialize rest of structure
  p_infilter_list->af = af; //address family of this list must match the address family of the owning port

  return p_infilter_list;
}

//free a input filter list
void infilter_free( st_infilter_list * p_infilter_list )
{
  //free list
  olist_free( p_infilter_list->p_olist, infilter_data_free );

  //free structure
  free( p_infilter_list );
}

//add a input filter rule to a input filter list
//returns 0 on success, a negative value in case of error
int infilter_add( st_infilter_list * p_infilter_list, char * p_index, char * p_range, char * p_proto, char * p_format, char * p_name, char stop, char * p_op_name, char * p_priority, char * p_timeout, st_out_info * p_out_info )
{
  unsigned int index;
  unsigned long value;
  char * p_chr;
  st_infilter_data * p_data;
  int ret_val;
  char txt[11];

  //convert index
  value = strtoul( p_index, &p_chr, 0 );
  index = (unsigned int)value;
  if( *p_index == 0 || *p_chr != 0 || index != value )
  {
    out_str( p_out_info, "  error: invalid index \"" );
    out_str( p_out_info, p_index );
    out_str( p_out_info, "\"\n" );
    return -EINVAL;
  }

  //allocate data item
  p_data = infilter_data_new( );
  if( p_data == NULL )
  {
    out_str( p_out_info, "  error: could allocate data item (out of memory)\n" );
    return -ENOMEM;
  }

  //convert address range
  if( addr_str2range( p_range, &p_data->range, p_out_info ) != 0 )
  {
    infilter_data_free( p_data );
    out_str( p_out_info, "  error: could not parse address range\n" );
    return -EINVAL;
  }
  //check that address family matches
  if( p_data->range.af != p_infilter_list->af )
  {
    infilter_data_free( p_data );
    out_str( p_out_info, "  error: address family does not match\n" );
    return -EINVAL;
  }

  //convert protocol
  p_data->proto = proto_str2proto( p_proto, 1 /* proto_any is allowed */, p_out_info );
  if( p_data->proto == proto_none )
  {
    infilter_data_free( p_data );
    out_str( p_out_info, "  error: could not parse protocol\n" );
    return -EINVAL;
  }

  //convert format
  if( format_str2format( p_format, &p_data->format, p_out_info ) != 0 )
  {
    infilter_data_free( p_data );
    out_str( p_out_info, "  error: could not parse format\n" );
    return -EINVAL;
  }

  //get (and truncate) name
  if( strlen( p_name ) >= list_item_name_size )
    p_name[list_item_name_size-1] = 0;
  strcpy( p_data->name, p_name );

  //get stop flag
  p_data->stop = stop != 0;

  //get (and truncate) output port name
  if( strlen( p_op_name ) >= list_item_name_size )
    p_op_name[list_item_name_size-1] = 0;
  strcpy( p_data->op_name, p_op_name );

  //convert priority
  value = strtoul( p_priority, &p_chr, 0 );
  p_data->priority = (unsigned short)value;
  if( *p_priority == 0 || *p_chr != 0 || p_data->priority != value )
  {
    infilter_data_free( p_data );
    out_str( p_out_info, "  error: invalid priority \"" );
    out_str( p_out_info, p_priority );
    out_str( p_out_info, "\"\n" );
    return -EINVAL;
  }

  //convert timeout
  value = strtoul( p_timeout, &p_chr, 0 );
  p_data->timeout = (unsigned short)value;
  if( *p_timeout == 0 || *p_chr != 0 || p_data->timeout != value )
  {
    infilter_data_free( p_data );
    out_str( p_out_info, "  error: invalid timeout \"" );
    out_str( p_out_info, p_timeout );
    out_str( p_out_info, "\"\n" );
    return -EINVAL;
  }

  //add input filter to list
  ret_val = olist_add( p_infilter_list->p_olist, index, p_data );
  if( ret_val != 0 )
  {
    infilter_data_free( p_data );
    out_str( p_out_info, "  error: could not add input filter rule (no. " );
    sprintf( txt, "%u", index );
    out_str( p_out_info, txt );
    out_str( p_out_info, ") to list" );
    switch( ret_val )
    {
      case -ENOMEM: out_str( p_out_info, " (out of memory)" ); break;
    }
    out_str( p_out_info, "\n" );
    return ret_val;
  }

  return 0;
}

//delete a input filter rule from a input filter list
//returns 0 on success, a negative value in case of error
int infilter_del( st_infilter_list * p_infilter_list, char * p_index, st_out_info * p_out_info )
{
  unsigned int index;
  unsigned long value;
  char * p_chr;
  int ret_val;
  char txt[11];
  
  //convert index
  value = strtoul( p_index, &p_chr, 0 );
  index = (unsigned int)value;
  if( *p_index == 0 || *p_chr != 0 || index != value )
  {
    out_str( p_out_info, "  error: invalid index \"" );
    out_str( p_out_info, p_index );
    out_str( p_out_info, "\"\n" );
    return -EINVAL;
  }

  //delete input filter from list
  ret_val = olist_del( p_infilter_list->p_olist, index, infilter_data_free );
  if( ret_val != 0 )
  {
    out_str( p_out_info, "  error: could not delete input filter rule (no. " );
    sprintf( txt, "%u", index );
    out_str( p_out_info, txt );
    out_str( p_out_info, ") from list" );
    switch( ret_val )
    {
      case -ENOENT: out_str( p_out_info, " (not in list)" ); break;
    }
    out_str( p_out_info, "\n" );
    return ret_val;
  }

  return 0;
}

//list input filter rules - enumeration function
static int infilter_list_enum( unsigned int index, void * vp_data, void * vp_out_info )
{
  st_infilter_data * p_data = vp_data;
  st_out_info * p_out_info = vp_out_info;
  char txt[15];

  //output index
  sprintf( txt, "  %u:\n", index );
  out_str( p_out_info, txt );

  //output address range and name
  out_str( p_out_info, "    address range: " );
  addr_range_out( &p_data->range, p_out_info );
  out_str( p_out_info, "\n    protocol: " );
  proto_proto_out( p_data->proto, p_out_info );
  out_str( p_out_info, "\n    format: " );
  format_format_out( &p_data->format, p_out_info );
  out_str( p_out_info, "\n    stream name: " );
  out_str( p_out_info, p_data->name );
  out_str( p_out_info, "\n" );

  //output stop flag
  if( p_data->stop )
    out_str( p_out_info, "    stop after this rule\n" );

  //output output port name, priority and timeout
  if( p_data->op_name[0] != 0 )
  {
    out_str( p_out_info, "    output port name: " );
    out_str( p_out_info, p_data->op_name );
    out_str( p_out_info, "\n    priority: " );
    sprintf( txt, "%u\n", p_data->priority );
    out_str( p_out_info, txt );
    out_str( p_out_info, "\n    timeout: " );
    sprintf( txt, "%u\n", p_data->timeout );
    out_str( p_out_info, txt );
  }

  //continue with enumeration
  return 0;
}

//list input filter rules in a input filter list
void infilter_list( st_infilter_list * p_infilter_list, st_out_info * p_out_info )
{
  //enumerate list items
  olist_enum( p_infilter_list->p_olist, infilter_list_enum, (void *)p_out_info );
}

//pass frame along input filter list - enumeration function
static int infilter_process_enum( unsigned int index, void * vp_data, void * vp_frame )
{
  st_infilter_data * p_data = vp_data;
  st_frame * p_frame = vp_frame;
  st_outport_data * p_outport_data;
  st_stream_list * p_stream_list;

  //check if frame matches input filter
  if( frame_match( p_frame, &p_data->range, p_data->proto, &p_data->format ) )
  {
    //submit this frame to the output port

    //get output port
    p_outport_data = outport_get( p_global_outport_list, p_data->op_name, p_out_none );
    if( p_outport_data != NULL )
    {
      //get stream list of output port
      p_stream_list = outport_stream_list( p_outport_data );
      if( p_stream_list != NULL )
        //submit frame to stream list
        stream_frame( p_stream_list, p_data->name, p_frame, p_data->priority, p_data->timeout );
    }

    //stop enumeration, if requested
    if( p_data->stop )
      return 1;
  }

  //continue with enumeration
  return 0;

  //keep compiler happy
  index = 0;
}

//pass frame along input filter list
void infilter_process( st_infilter_list * p_infilter_list, st_frame * p_frame )
{
  //enumerate list items
  olist_enum( p_infilter_list->p_olist, infilter_process_enum, (void *)p_frame );
}
