/* BLINKENLIGHTS PROXY
 * version 0.98 date 2006-07-27
 * Copyright (C) 2003-2006 Stefan Schuermans <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 */

#include <stdlib.h>
#include <errno.h>

#include "addr.h"
#include "addr_list.h"

//type for a list item
typedef struct s_addr_list_item
{
  st_addr src_addr, dest_addr; //addresses of this item
  void * p_data; //data of this item
} st_addr_list_item;

//type for an address list (see header file)
struct s_addr_list
{
  unsigned int len; //current length of the list (measured in entries)
  unsigned int size; //current size of the list (measured in entries)
  unsigned int size_step; //current size step of the list (measured in entries)
  st_addr_list_item * * p_list; //pointer to array containing list
};

//get position for/of entry
static void addr_list_get_pos( st_addr_list_item * * p_list, unsigned int len, st_addr * p_src_addr, st_addr * p_dest_addr, int * p_start, int * p_end )
{
  int i;
  int cmp;

  //empty list
  if( len == 0 )
  {
    //insert at position 0
    *p_start = -1;
    *p_end = 0;
    return;
  }

  //start with entire range
  *p_start = 0;
  *p_end = (int)len - 1;

  //do a binary search
  for( ; ; )
  {
    //get middle of range
    i = (*p_start + *p_end) / 2;
    //compare
    cmp = addr_cmp( p_src_addr, &p_list[i]->src_addr );
    if( cmp == 0 )
      cmp = addr_cmp( p_dest_addr, &p_list[i]->dest_addr );
    //searched item is above middle
    if( cmp < 0 )
    {
      //not found
      if( i <= *p_start )
      {
        //insert at position i
        *p_start = i - 1;
        *p_end = i;
        break;
      }
      //search above middle
      *p_end = i - 1;
    }
    //searched item is below middle
    else if( cmp > 0 )
    {
      //not found
      if( i >= *p_end )
      {
        //insert at position i + 1
        *p_start = i;
        *p_end = i + 1;
        break;
      }
      //search in below middle
      *p_start = i + 1;
    }
    //found
    else
    {
      //found at position i
      *p_start = i;
      *p_end = i;
      break;
    }
  }
}

//create a new address list
//returns NULL in case of error
st_addr_list * addr_list_new( )
{
  st_addr_list * p_addr_list;

  //allocate memeory for list head
  p_addr_list = (st_addr_list *)malloc( sizeof( st_addr_list ) );
  if( p_addr_list == NULL )
    return NULL;

  //initialize list
  p_addr_list->len = 0; //list is empty
  p_addr_list->size = 16; //start with space for some entries
  p_addr_list->size_step = 4; //increase size in steps of quarter to almost half size
  p_addr_list->p_list = (st_addr_list_item * *)malloc( p_addr_list->size * sizeof( st_addr_list_item * ) );
  if( p_addr_list->p_list == NULL )
  {
    free( p_addr_list );
    return NULL;
  }

  return p_addr_list;
}

//delete an address list (including all entries)
//data_free_func is called for the data of every list item (if not NULL)
void addr_list_free( st_addr_list * p_addr_list, void (*data_free_func)( void * ) )
{
  unsigned int i;

  //free list entries
  for( i = 0; i < p_addr_list->len; i++ )
  {
    //free data
    if( p_addr_list->p_list[i]->p_data != NULL && data_free_func != NULL )
      data_free_func( p_addr_list->p_list[i]->p_data );
    //free list entry
    free( p_addr_list->p_list[i] );
  }

  //free array and list head
  free( p_addr_list->p_list );
  free( p_addr_list );
}

//add an item to an address list
//returns 0 on success, a negative value on error
int addr_list_add( st_addr_list * p_addr_list, st_addr * p_src_addr, st_addr * p_dest_addr, void * p_data )
{
  int start, end, i;
  st_addr_list_item * p_item, * * p_new_list;

  //get position
  addr_list_get_pos( p_addr_list->p_list, p_addr_list->len, p_src_addr, p_dest_addr, &start, &end );
  //already in list
  if( start == end )
    return -EEXIST;

  //allocate list item
  p_item = (st_addr_list_item *)malloc( sizeof( st_addr_list_item ) );
  if( p_item == NULL )
    return -ENOMEM;

  //initialize item
  p_item->src_addr = *p_src_addr;
  p_item->dest_addr = *p_dest_addr;
  p_item->p_data = p_data;

  //increase length of list
  p_addr_list->len++;

  //must size be increased?
  if( p_addr_list->len > p_addr_list->size )
  {
    //must size step be increased?
    if( p_addr_list->size >= p_addr_list->size_step << 2 )
      p_addr_list->size_step <<= 1;
    //increase size
    p_addr_list->size += p_addr_list->size_step;
    //reallocate list
    p_new_list = (st_addr_list_item * *)realloc( p_addr_list->p_list, p_addr_list->size * sizeof( st_addr_list_item * ) );
    if( p_new_list == NULL )
    {
      free( p_item );
      return -ENOMEM;
    }
    p_addr_list->p_list = p_new_list;
  }

  //put new item into list
  for( i = p_addr_list->len - 1; i > end; i-- )
    p_addr_list->p_list[i] = p_addr_list->p_list[i-1];
  p_addr_list->p_list[end] = p_item;

  return 0;
}

//delete an item from an address list
//data_free_func is called for the data of deleted list item (if not NULL)
//returns 0 on success, a negative value on error
int addr_list_del( st_addr_list * p_addr_list, st_addr * p_src_addr, st_addr * p_dest_addr, void (*data_free_func)( void * ) )
{
  int start, end, i;
  st_addr_list_item * * p_new_list;

  //get position
  addr_list_get_pos( p_addr_list->p_list, p_addr_list->len, p_src_addr, p_dest_addr, &start, &end );
  //not in list
  if( start != end )
    return -ENOENT;

  //free data
  if( p_addr_list->p_list[end]->p_data != NULL && data_free_func != NULL )
    data_free_func( p_addr_list->p_list[end]->p_data );
  //free list entry
  free( p_addr_list->p_list[end] );

  //decrease length of list
  p_addr_list->len--;
  
  //remove item from list
  for( i = end; i < (int)p_addr_list->len; i++ )
    p_addr_list->p_list[i] = p_addr_list->p_list[i+1];

  //can size be decreased?
  if( p_addr_list->len + p_addr_list->size_step <= p_addr_list->size )
  {
    //decrease size
    p_addr_list->size -= p_addr_list->size_step;
    if( p_addr_list->size < 16 ) //keep a minimum size of 16 entries
      p_addr_list->size = 16;
    //reallocate list
    p_new_list = (st_addr_list_item * *)realloc( p_addr_list->p_list, p_addr_list->size * sizeof( st_addr_list_item * ) );
    if( p_new_list != NULL ) //if realloc failed, we keep the bigger buffer
      p_addr_list->p_list = p_new_list;
    //must size step be decreased?
    if( p_addr_list->size <= p_addr_list->size_step << 1 )
    {
      p_addr_list->size_step >>= 1;
      if( p_addr_list->size_step < 4 ) //keep a minimum size step of 4 entries
        p_addr_list->size_step = 4;
    }
  }

  return 0;
}

//get the data of an item in an address list
//returns the data pointer of NULL on error
void * addr_list_get( st_addr_list * p_addr_list, st_addr * p_src_addr, st_addr * p_dest_addr )
{
  int start, end;

  //get position
  addr_list_get_pos( p_addr_list->p_list, p_addr_list->len, p_src_addr, p_dest_addr, &start, &end );
  //not in list
  if( start != end )
    return NULL;

  //return data pointer
  return p_addr_list->p_list[end]->p_data;
}

//enumerate all items in an address list
//the enumerations stops when enum_func retuns a value != 0
void addr_list_enum( st_addr_list * p_addr_list, int (*enum_func)( st_addr *, st_addr *, void *, void * ), void * p_enum_context )
{
  int i;

  //call enumeration function for every item in the list
  for( i = 0; i < (int)p_addr_list->len; i++ )
    if( enum_func( &p_addr_list->p_list[i]->src_addr, &p_addr_list->p_list[i]->dest_addr, p_addr_list->p_list[i]->p_data, p_enum_context ) != 0 )
      break;
}

//delete some items in an address list by enumerating
//the entry is delted from the list if enum_func retuns a value != 0
void addr_list_enum_del( st_addr_list * p_addr_list, int (*enum_func)( st_addr *, st_addr *, void *, void * ), void * p_enum_context, void (*data_free_func)( void * ) )
{
  int i, j;
  st_addr_list_item * * p_new_list;

  //call enumeration function for every item in the list
  for( i = 0; i < (int)p_addr_list->len; i++ )
  {
    //check if item shall be deleted
    if( enum_func( &p_addr_list->p_list[i]->src_addr, &p_addr_list->p_list[i]->dest_addr, p_addr_list->p_list[i]->p_data, p_enum_context ) != 0 )
    {
      //free data
      if( p_addr_list->p_list[i]->p_data != NULL && data_free_func != NULL )
        data_free_func( p_addr_list->p_list[i]->p_data );
      //free list entry
      free( p_addr_list->p_list[i] );

      //decrease length of list
      p_addr_list->len--;
      
      //remove item from list
      for( j = i; j < (int)p_addr_list->len; j++ )
        p_addr_list->p_list[j] = p_addr_list->p_list[j+1];

      //use current index again in next turn of loop
      i--;
    }
  }

  //can size be decreased?
  if( p_addr_list->len + p_addr_list->size_step <= p_addr_list->size )
  {
    //decrease size
    p_addr_list->size -= p_addr_list->size_step;
    if( p_addr_list->size < 16 ) //keep a minimum size of 16 entries
      p_addr_list->size = 16;
    //reallocate list
    p_new_list = (st_addr_list_item * *)realloc( p_addr_list->p_list, p_addr_list->size * sizeof( st_addr_list_item * ) );
    if( p_new_list != NULL ) //if realloc failed, we keep the bigger buffer
      p_addr_list->p_list = p_new_list;
    //must size step be decreased?
    if( p_addr_list->size <= p_addr_list->size_step << 1 )
    {
      p_addr_list->size_step >>= 1;
      if( p_addr_list->size_step < 4 ) //keep a minimum size step of 4 entries
        p_addr_list->size_step = 4;
    }
  }
}
