/* BLINKENLIGHTS PROXY
 * version 0.97 date 2005-12-29
 * Copyright (C) 2003-2005 Stefan Schuermans <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>

#include "addr.h"
#include "list.h"
#include "olist.h"
#include "outport.h"
#include "output.h"
#include "proto.h"
#include "outfilter.h"

//type for data of output filter list items (see header file)
struct s_outfilter_data
{
  st_addr_range range; //the address range to match
  et_proto proto; //the protocol to match
  char name[list_item_name_size]; //name for the destination
  char deny; //!=0: deny the access to the stream
  unsigned short timeout; //timeout (in seconds)
  st_format format; //the format to output in
};

//type for an output filter list (see header file)
struct s_outfilter_list
{
  st_olist * p_olist; //the output filter list
  int af; //the address family of this list (obtained from owning port)
  st_outport_data * p_outport_data; //pointer to owning outport
};

//create a new output filter list data item
//returns the pointer to the data item or NULL in case of error
static st_outfilter_data * outfilter_data_new( )
{
  st_outfilter_data * p_data;

  //allocate a new output filter list data item
  p_data = (st_outfilter_data *)malloc( sizeof( st_outfilter_data ) );
  if( p_data == NULL )
    return NULL;

  //initialize data item
  p_data->range.af = AF_UNSPEC; //no address range yet
  p_data->proto = proto_none; //no protocol yet
  p_data->name[0] = 0; //no name yet
  p_data->deny = 0; //do not deny access to the stream
  p_data->timeout = 0; //immediate timeout
  p_data->format = format_none; //no format yet

  return p_data;
}

//free output filter list data item
static void outfilter_data_free( void * vp_data )
{
  st_outfilter_data * p_data = vp_data;

  //free data item
  free( p_data );
}

//create a new output filter list
//<af> specifies the address family of this output filter list (obtained from owning port)
//returns the pointer to the list or NULL in case of error
st_outfilter_list * outfilter_new( int af, st_out_info * p_out_info, st_outport_data * p_outport_data )
{
  st_outfilter_list * p_outfilter_list;

  //create structure
  p_outfilter_list = (st_outfilter_list *)malloc( sizeof( st_outfilter_list ) );
  if( p_outfilter_list == NULL )
  {
    out_str( p_out_info, "  error: could not create new output filter list (out of memory)\n" );
    return NULL;
  }

  //create list
  p_outfilter_list->p_olist = olist_new( );
  if( p_outfilter_list->p_olist == NULL )
  {
    free( p_outfilter_list );
    out_str( p_out_info, "  error: could not create new output filter list\n" );
    return NULL;
  }

  //initialize rest of structure
  p_outfilter_list->af = af; //address family of this list must match the address family of the owning port
  p_outfilter_list->p_outport_data = p_outport_data; //pointer to owning output port

  return p_outfilter_list;
}

//free an output filter list
void outfilter_free( st_outfilter_list * p_outfilter_list )
{
  //free list
  olist_free( p_outfilter_list->p_olist, outfilter_data_free );

  //free structure
  free( p_outfilter_list );
}

//add an output filter rule to a output filter list
//returns 0 on success, a negative value in case of error
int outfilter_add( st_outfilter_list * p_outfilter_list, char * p_index, char * p_range, char * p_proto, char * p_name, char deny, char * p_timeout, char * p_format, st_out_info * p_out_info )
{
  unsigned int index;
  unsigned long value;
  char * p_chr;
  st_outfilter_data * p_data;
  int ret_val;
  char txt[11];

  //convert index
  value = strtoul( p_index, &p_chr, 0 );
  index = (unsigned int)value;
  if( *p_index == 0 || *p_chr != 0 || index != value )
  {
    out_str( p_out_info, "  error: invalid index \"" );
    out_str( p_out_info, p_index );
    out_str( p_out_info, "\"\n" );
    return -EINVAL;
  }

  //allocate data item
  p_data = outfilter_data_new( );
  if( p_data == NULL )
  {
    out_str( p_out_info, "  error: could allocate data item (out of memory)\n" );
    return -ENOMEM;
  }

  //convert address range
  if( addr_str2range( p_range, &p_data->range, p_out_info ) != 0 )
  {
    outfilter_data_free( p_data );
    out_str( p_out_info, "  error: could not parse address range\n" );
    return -EINVAL;
  }
  //check that address family matches
  if( p_data->range.af != p_outfilter_list->af )
  {
    outfilter_data_free( p_data );
    out_str( p_out_info, "  error: address family does not match\n" );
    return -EINVAL;
  }

  //convert protocol
  p_data->proto = proto_str2proto( p_proto, 1 /* proto_any is allowed */, p_out_info );
  if( p_data->proto == proto_none )
  {
    outfilter_data_free( p_data );
    out_str( p_out_info, "  error: could not parse protocol\n" );
    return -EINVAL;
  }

  //get (and truncate) name
  if( strlen( p_name ) >= list_item_name_size )
    p_name[list_item_name_size-1] = 0;
  strcpy( p_data->name, p_name );

  //get deny flag
  p_data->deny = deny != 0;

  //convert timeout
  if( ! p_data->deny )
  {
    value = strtoul( p_timeout, &p_chr, 0 );
    p_data->timeout = (unsigned short)value;
    if( *p_timeout == 0 || *p_chr != 0 || p_data->timeout != value || p_data->timeout == 0 )
    {
      outfilter_data_free( p_data );
      out_str( p_out_info, "  error: invalid timeout \"" );
      out_str( p_out_info, p_timeout );
      out_str( p_out_info, "\"\n" );
      return -EINVAL;
    }
  }

  //convert format
  if( format_str2format( p_format, &p_data->format, p_out_info ) != 0 )
  {
    outfilter_data_free( p_data );
    out_str( p_out_info, "  error: could not parse format\n" );
    return -EINVAL;
  }

  //add output filter to list
  ret_val = olist_add( p_outfilter_list->p_olist, index, p_data );
  if( ret_val != 0 )
  {
    outfilter_data_free( p_data );
    out_str( p_out_info, "  error: could not add output filter rule (no. " );
    sprintf( txt, "%u", index );
    out_str( p_out_info, txt );
    out_str( p_out_info, ") to list" );
    switch( ret_val )
    {
      case -ENOMEM: out_str( p_out_info, " (out of memory)" ); break;
    }
    out_str( p_out_info, "\n" );
    return ret_val;
  }

  return 0;
}

//delete an output filter rule from a output filter list
//returns 0 on success, a negative value in case of error
int outfilter_del( st_outfilter_list * p_outfilter_list, char * p_index, st_out_info * p_out_info )
{
  unsigned int index;
  unsigned long value;
  char * p_chr;
  int ret_val;
  char txt[11];
  
  //convert index
  value = strtoul( p_index, &p_chr, 0 );
  index = (unsigned int)value;
  if( *p_index == 0 || *p_chr != 0 || index != value )
  {
    out_str( p_out_info, "  error: invalid index \"" );
    out_str( p_out_info, p_index );
    out_str( p_out_info, "\"\n" );
    return -EINVAL;
  }

  //delete output filter from list
  ret_val = olist_del( p_outfilter_list->p_olist, index, outfilter_data_free );
  if( ret_val != 0 )
  {
    out_str( p_out_info, "  error: could not delete output filter rule (no. " );
    sprintf( txt, "%u", index );
    out_str( p_out_info, txt );
    out_str( p_out_info, ") from list" );
    switch( ret_val )
    {
      case -ENOENT: out_str( p_out_info, " (not in list)" ); break;
    }
    out_str( p_out_info, "\n" );
    return ret_val;
  }

  return 0;
}

//list output filter rules - enumeration function
static int outfilter_list_enum( unsigned int index, void * vp_data, void * vp_out_info )
{
  st_outfilter_data * p_data = vp_data;
  st_out_info * p_out_info = vp_out_info;
  char txt[15];

  //output index
  sprintf( txt, "  %u:\n", index );
  out_str( p_out_info, txt );

  //output address range and name
  out_str( p_out_info, "    address range: " );
  addr_range_out( &p_data->range, p_out_info );
  out_str( p_out_info, "\n    protocol: " );
  proto_proto_out( p_data->proto, p_out_info );
  out_str( p_out_info, "\n    stream name: " );
  out_str( p_out_info, p_data->name );
  out_str( p_out_info, "\n" );

  //access denied
  if( p_data->deny )
    out_str( p_out_info, "    deny\n" );
  //access allowed
  else
  {
    //output timeout
    out_str( p_out_info, "    timeout: " );
    sprintf( txt, "%u s\n", p_data->timeout );
    out_str( p_out_info, txt );
  }
  out_str( p_out_info, "    format: " );
  format_format_out( &p_data->format, p_out_info );
  out_str( p_out_info, "\n" );

  //continue with enumeration
  return 0;
}

//list output filter rules in an output filter list
void outfilter_list( st_outfilter_list * p_outfilter_list, st_out_info * p_out_info )
{
  //enumerate list items
  olist_enum( p_outfilter_list->p_olist, outfilter_list_enum, (void *)p_out_info );
}

//pass request along output filter list - type for enumeration context
typedef struct s_outfilter_process_context
{
  st_outport_data * p_outport_data;
  st_addr * p_addr;
  et_proto proto;
  int enable;
} st_outfilter_process_context;

//pass request along output filter list - enumeration function
static int outfilter_process_enum( unsigned int index, void * vp_data, void * vp_context )
{
  st_outfilter_data * p_data = vp_data;
  st_outfilter_process_context * p_context = vp_context;
  st_dyndest_list * p_dyndest_list;
  st_addr * p_outport_addr;

  //check if request matches output filter
  if( proto_request_match( p_context->p_addr, p_context->proto, &p_data->range, p_data->proto ) )
  {
    //stop enumeration, if access is denied
    if( p_data->deny )
      return 1;

    //get dynamic destination list and address of output port
    p_dyndest_list = outport_dyndest_list( p_context->p_outport_data );
    p_outport_addr = outport_addr( p_context->p_outport_data );

    //add dynamic destination
    if( p_context->enable )
      dyndest_add( p_dyndest_list, p_outport_addr, p_context->p_addr, p_data->name, p_context->proto, &p_data->format, p_data->timeout );
    //remove dynamic destination
    else
      dyndest_del( p_dyndest_list, p_outport_addr, p_context->p_addr );
  }

  //continue with enumeration
  return 0;

  //keep compiler happy
  index = 0;
}

//pass request along output filter list
void outfilter_process( st_outfilter_list * p_outfilter_list, st_addr * p_addr, et_proto proto, int enable )
{
  st_outfilter_process_context context;

  //enumerate list items
  context.p_outport_data = p_outfilter_list->p_outport_data;
  context.p_addr = p_addr;
  context.proto = proto;
  context.enable = enable;
  olist_enum( p_outfilter_list->p_olist, outfilter_process_enum, (void *)&context );
}
