/* BLINKENLIGHTS PROXY
 * version 0.95 date 2004-09-30
 * Copyright (C) 2003-2004 1stein <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <malloc.h>
#include <sys/types.h>
#include <sys/socket.h>

#include "addr.h"
#include "global.h"
#include "list.h"
#include "output.h"
#include "proto.h"
#include "dynsrc.h"

//type for data of dynamic source list items (see header file)
struct s_dynsrc_data
{
  st_addr addr; //the address to sent the requests to
  et_proto proto; //protocol to use
  unsigned short interval; //interval for sending requests (in seconds)
  unsigned long last_millisecs; //milliseconds when last request was sent
};

//type for a dynamic source list (see header file)
struct s_dynsrc_list
{
  st_list * p_list; //the dynamic source list
  int af; //the address family of this list (obtained from owning port)
};

//create a new dynamic source list data item
//returns the pointer to the data item or NULL in case of error
static st_dynsrc_data * dynsrc_data_new( )
{
  st_dynsrc_data * p_data;

  //allocate a new dynamic source list data item
  p_data = (st_dynsrc_data *)malloc( sizeof( st_dynsrc_data ) );
  if( p_data == NULL )
    return NULL;

  //initialize data item
  p_data->addr.af = AF_UNSPEC; //no address yet
  p_data->addr.sa_len = 0;
  p_data->proto = proto_none; //no protocol yet
  p_data->interval = 0; //no interval yet

  return p_data;
}

//free dynamic source list data item
static void dynsrc_data_free( void * vp_data )
{
  st_dynsrc_data * p_data = vp_data;

  //free data item
  free( p_data );
}

//create a new dynamic source list
//<af> specifies the address family of this dynamic source list (obtained from owning port)
//returns the pointer to the list or NULL in case of error
st_dynsrc_list * dynsrc_new( int af, st_out_info * p_out_info )
{
  st_dynsrc_list * p_dynsrc_list;

  //create structure
  p_dynsrc_list = (st_dynsrc_list *)malloc( sizeof( st_dynsrc_list ) );
  if( p_dynsrc_list == NULL )
  {
    out_str( p_out_info, "  error: could not create new dynamic source list (out of memory)\n" );
    return NULL;
  }

  //create list
  p_dynsrc_list->p_list = list_new( );
  if( p_dynsrc_list->p_list == NULL )
  {
    free( p_dynsrc_list );
    out_str( p_out_info, "  error: could not create new dynamic source list\n" );
    return NULL;
  }

  //initialize rest of structure
  p_dynsrc_list->af = af; //address family of this list must match the address family of the owning port

  return p_dynsrc_list;
}

//free a dynamic source list
void dynsrc_free( st_dynsrc_list * p_dynsrc_list )
{
  //free list
  list_free( p_dynsrc_list->p_list, dynsrc_data_free );

  //free structure
  free( p_dynsrc_list );
}

//add a dynamic source to a dynamic source list
//returns 0 on success, a negative value in case of error
int dynsrc_add( st_dynsrc_list * p_dynsrc_list, char * p_name, char * p_addr, char * p_proto, char * p_interval, int sock, st_out_info * p_out_info )
{
  st_dynsrc_data * p_data;
  char * p_chr;
  unsigned long value;
  int ret_val;

  //allocate data item
  p_data = dynsrc_data_new( );
  if( p_data == NULL )
  {
    out_str( p_out_info, "  error: could allocate data item (out of memory)\n" );
    return -ENOMEM;
  }

  //convert address
  if( addr_str2addr( p_addr, &p_data->addr, p_out_info ) != 0 )
  {
    dynsrc_data_free( p_data );
    out_str( p_out_info, "  error: could not parse address\n" );
    return -EINVAL;
  }
  //check that address family matches
  if( p_data->addr.af != p_dynsrc_list->af )
  {
    dynsrc_data_free( p_data );
    out_str( p_out_info, "  error: address family does not match\n" );
    return -EINVAL;
  }

  //convert protocol
  p_data->proto = proto_str2proto( p_proto, 0 /* proto_any is not allowed */, p_out_info );
  if( p_data->proto == proto_none )
  {
    dynsrc_data_free( p_data );
    out_str( p_out_info, "  error: could not parse protocol\n" );
    return -EINVAL;
  }

  //convert interval
  value = strtoul( p_interval, &p_chr, 0 );
  p_data->interval = (unsigned short)value;
  if( *p_interval == 0 || *p_chr != 0 || p_data->interval != value || p_data->interval == 0 )
  {
    dynsrc_data_free( p_data );
    out_str( p_out_info, "  error: invalid interval \"" );
    out_str( p_out_info, p_interval );
    out_str( p_out_info, "\"\n" );
    return -EINVAL;
  }

  //add dynamic source to list
  ret_val = list_add( p_dynsrc_list->p_list, p_name, p_data );
  if( ret_val != 0 )
  {
    dynsrc_data_free( p_data );
    out_str( p_out_info, "  error: could not add dynamic source \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\" to list" );
    switch( ret_val )
    {
      case -EEXIST: out_str( p_out_info, " (already in list)" ); break;
      case -ENOMEM: out_str( p_out_info, " (out of memory)" ); break;
    }
    out_str( p_out_info, "\n" );
    return ret_val;
  }

  //send dynaic stream refresh
  proto_send_refresh( sock, p_data->proto, &p_data->addr );
  //remember milliseconds
  p_data->last_millisecs = global_millisecs;  

  return 0;
}

//delete a dynamic source from a dynamic source list
//returns 0 on success, a negative value in case of error
int dynsrc_del( st_dynsrc_list * p_dynsrc_list, char * p_name, int sock, st_out_info * p_out_info )
{
  st_dynsrc_data * p_data;
  int ret_val;
  
  //get dynamic source list item
  p_data = list_get( p_dynsrc_list->p_list, p_name );
  if( p_data != NULL )
    //send dynamic stream close
    proto_send_close( sock, p_data->proto, &p_data->addr );
  else
  {
    out_str( p_out_info, "  warning: stream will not be closed: could not find dynamic source \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\" in list" );
    out_str( p_out_info, "\n" );
  }

  //delete dynamic source from list
  ret_val = list_del( p_dynsrc_list->p_list, p_name, dynsrc_data_free );
  if( ret_val != 0 )
  {
    out_str( p_out_info, "  error: could not delete dynamic source \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\" from list" );
    switch( ret_val )
    {
      case -ENOENT: out_str( p_out_info, " (not in list)" ); break;
    }
    out_str( p_out_info, "\n" );
    return ret_val;
  }

  return 0;
}

//list dynamic sources - enumeration function
static int dynsrc_list_enum( char * p_name, void * vp_data, void * vp_out_info )
{
  st_dynsrc_data * p_data = vp_data;
  st_out_info * p_out_info = vp_out_info;
  char txt[11];

  //output name
  out_str( p_out_info, "  " );
  out_str( p_out_info, p_name );
  out_str( p_out_info, "\n" );

  //output address, protocol and timeout
  out_str( p_out_info, "    address: " );
  addr_addr_out( &p_data->addr, p_out_info );
  out_str( p_out_info, "\n    protocol: " );
  proto_proto_out( p_data->proto, p_out_info );
  out_str( p_out_info, "\n    interval: " );
  sprintf( txt, "%u s\n", p_data->interval );
  out_str( p_out_info, txt );

  //continue with enumeration
  return 0;
}

//list dynamic sources in a dynamic source list
void dynsrc_list( st_dynsrc_list * p_dynsrc_list, st_out_info * p_out_info )
{
  //enumerate list items
  list_enum( p_dynsrc_list->p_list, dynsrc_list_enum, (void *)p_out_info );
}

//do periodic tasks for dynamic sources - enumeration function
static int dynsrc_tick_enum( char * p_name, void * vp_data, void * vp_sock )
{
  st_dynsrc_data * p_data = vp_data;
  int * p_sock = vp_sock;

  //interval for dynamic source refresh passed
  if( (long)(global_millisecs - p_data->last_millisecs) > (long)p_data->interval * 1000 )
  {
    //send dynaic stream refresh
    proto_send_refresh( *p_sock, p_data->proto, &p_data->addr );
    //remember milliseconds
    p_data->last_millisecs = global_millisecs;  
  }

  //continue with enumeration
  return 0;

  //keep compiler happy
  p_name = NULL;
}

//do periodic tasks for dynamic sources
void dynsrc_tick( st_dynsrc_list * p_dynsrc_list, int sock )
{
  //enumerate list items
  list_enum( p_dynsrc_list->p_list, dynsrc_tick_enum, (void *)&sock );
}
