/* BLINKENLIGHTS PROXY
 * version 0.94 date 2004-06-07
 * Copyright (C) 2003-2004 1stein <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 */

#include <malloc.h>
#include <errno.h>

#include "olist.h"

//type for a list (see header file)
struct s_olist
{
  unsigned int len; //current length of the list (measured in entries)
  unsigned int size; //current size of the list (measured in entries)
  unsigned int size_step; //current size step of the list (measured in entries)
  void * * p_list; //pointer to array containing list
};

//create a new ordered list
//returns NULL in case of error
st_olist * olist_new( )
{
  st_olist * p_olist;

  //allocate memeory for list head
  p_olist = (st_olist *)malloc( sizeof( st_olist ) );
  if( p_olist == NULL )
    return NULL;

  //initialize list
  p_olist->len = 0; //list is empty
  p_olist->size = 16; //start with space for some entries
  p_olist->size_step = 4; //increase size in steps of quarter to almost half size
  p_olist->p_list = (void * *)malloc( p_olist->size * sizeof( void * ) );
  if( p_olist->p_list == NULL )
  {
    free( p_olist );
    return NULL;
  }

  return p_olist;
}

//delete an ordered list (including all entries)
//data_free_func is called for the data of every list item (if not NULL)
void olist_free( st_olist * p_olist, void (*data_free_func)( void * ) )
{
  unsigned int i;

  //free list entries
  for( i = 0; i < p_olist->len; i++ )
    if( p_olist->p_list[i] != NULL && data_free_func != NULL )
      data_free_func( p_olist->p_list[i] );

  //free array and list head
  free( p_olist->p_list );
  free( p_olist );
}

//add an item to an ordered list
//returns 0 on success, a negative value on error
int olist_add( st_olist * p_olist, unsigned int index, void * p_data )
{
  unsigned int i;
  void * * p_new_list;

  //correct index
  if( index > p_olist->len )
    index = p_olist->len;

  //increase length of list
  p_olist->len++;

  //must size be increased?
  if( p_olist->len > p_olist->size )
  {
    //must size step be increased?
    if( p_olist->size >= p_olist->size_step << 2 )
      p_olist->size_step <<= 1;
    //increase size
    p_olist->size += p_olist->size_step;
    //reallocate list
    p_new_list = (void * *)realloc( p_olist->p_list, p_olist->size * sizeof( void * ) );
    if( p_new_list == NULL )
      return -ENOMEM;
    p_olist->p_list = p_new_list;
  }

  //put new item into list
  for( i = p_olist->len - 1; i > index; i-- )
    p_olist->p_list[i] = p_olist->p_list[i-1];
  p_olist->p_list[index] = p_data;

  return 0;
}

//delete an item from an ordered list
//data_free_func is called for the data of deleted list item (if not NULL)
//returns 0 on success, a negative value on error
int olist_del( st_olist * p_olist, unsigned int index, void (*data_free_func)( void * ) )
{
  unsigned int i;
  void * * p_new_list;

  //check index (if not in list)
  if( index >= p_olist->len )
    return -ENOENT;

  //free data
  if( p_olist->p_list[index] != NULL && data_free_func != NULL )
    data_free_func( p_olist->p_list[index] );

  //decrease length of list
  p_olist->len--;
  
  //remove item from list
  for( i = index; i < p_olist->len; i++ )
    p_olist->p_list[i] = p_olist->p_list[i+1];

  //can size be decreased?
  if( p_olist->len + p_olist->size_step <= p_olist->size )
  {
    //decrease size
    p_olist->size -= p_olist->size_step;
    if( p_olist->size < 16 ) //keep a minimum size of 16 entries
      p_olist->size = 16;
    //reallocate list
    p_new_list = (void * *)realloc( p_olist->p_list, p_olist->size * sizeof( void * ) );
    if( p_new_list != NULL ) //if realloc failed, we keep the bigger buffer
      p_olist->p_list = p_new_list;
    //must size step be decreased?
    if( p_olist->size <= p_olist->size_step << 1 )
    {
      p_olist->size_step >>= 1;
      if( p_olist->size_step < 4 ) //keep a minimum size step of 4 entries
        p_olist->size_step = 4;
    }
  }

  return 0;
}

//get the data of an item in an ordered list
//returns the data pointer of NULL on error
void * olist_get( st_olist * p_olist, unsigned int index )
{
  //check index (if not in list)
  if( index >= p_olist->len )
    return NULL;

  //return data pointer
  return p_olist->p_list[index];
}

//enumerate all items in an ordered list
//the enumerations stops when enum_func retuns a value != 0
void olist_enum( st_olist * p_olist, int (*enum_func)( unsigned int, void *, void * ), void * p_enum_context )
{
  unsigned int i;

  //call enumeration function for every item in the list
  for( i = 0; i < p_olist->len; i++ )
    if( enum_func( i, p_olist->p_list[i], p_enum_context ) != 0 )
      break;
}
