/* BLINKENLIGHTS PROXY
 * version 0.93 date 2004-04-26
 * Copyright (C) 2003-2004 1stein <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <errno.h>

#include "addr_list.h"
#include "list.h"
#include "global.h"
#include "outport.h"
#include "output.h"
#include "proto.h"
#include "stream.h"

//type for data of stream list items (see header file)
struct s_stream_data
{
  char name[list_item_name_size]; //name of this stream
  st_frame * p_frame; //pointer to current frame of this stream
  unsigned short priority; //priority of this stream
  unsigned short timeout; //timeout (in seconds)
  unsigned long timeout_msec; //timeout (in milliseconds)
                              //sort of copy of <timeout>
			      //used to avoid multiplications during ticks
  unsigned long last_frame_msec; //msecs when last frame was received
};

//type for a stream list (see header file)
struct s_stream_list
{
  st_addr_list * p_list; //the stream list
  st_outport_data * p_outport_data; //pointer to owning outport
  st_stream_data * p_current; //pointer to current stream (may be NULL)
};

//create a new stream list data item
//returns the pointer to the data item or NULL in case of error
static st_stream_data * stream_data_new( )
{
  st_stream_data * p_data;

  //allocate a new stream list data item
  p_data = (st_stream_data *)malloc( sizeof( st_stream_data ) );
  if( p_data == NULL )
    return NULL;

  //initialize data item
  p_data->name[0] = 0; //no name yet
  p_data->p_frame = NULL; //no frame yet
  p_data->priority = 0; //lowest priority for now
  p_data->timeout = 0; //imediate timeout for now
  p_data->timeout_msec = 0;
  p_data->last_frame_msec = global_millisecs; //last frame was received now

  return p_data;
}

//free stream list data item
static void stream_data_free( void * vp_data )
{
  st_stream_data * p_data = vp_data;

  //free data item
  free( p_data );
}

//create a new stream list
//returns the pointer to the list or NULL in case of error
st_stream_list * stream_new( st_outport_data * p_outport_data, st_out_info * p_out_info )
{
  st_stream_list * p_stream_list;

  //create structure
  p_stream_list = (st_stream_list *)malloc( sizeof( st_stream_list ) );
  if( p_stream_list == NULL )
  {
    out_str( p_out_info, "  error: could not create new stream list (out of memory)\n" );
    return NULL;
  }

  //create list
  p_stream_list->p_list = addr_list_new( );
  if( p_stream_list->p_list == NULL )
  {
    free( p_stream_list );
    out_str( p_out_info, "  error: could not create new stream list\n" );
    return NULL;
  }

  //initialize rest of structure
  p_stream_list->p_outport_data = p_outport_data; //save pointer to owning outport
  p_stream_list->p_current = NULL; //no current stream yet

  return p_stream_list;
}

//free a stream list
void stream_free( st_stream_list * p_stream_list )
{
  //free list
  addr_list_free( p_stream_list->p_list, stream_data_free );

  //free structure
  free( p_stream_list );
}

//list streams - enumeration context
typedef struct s_stream_list_context
{
  st_stream_list * p_stream_list;
  st_out_info * p_out_info;
} st_stream_list_context;

//list streams - enumeration function
static int stream_list_enum( st_addr * p_src_addr, st_addr * p_dest_addr, void * vp_data, void * vp_context )
{
  st_stream_data * p_data = vp_data;
  st_stream_list_context * p_context = vp_context;
  unsigned long timeout_msec;
  char txt[32];

  //output name, source address and destination address
  out_str( p_context->p_out_info, "  " );
  out_str( p_context->p_out_info, p_data->name );
  out_str( p_context->p_out_info, ": " );
  addr_addr_out( p_src_addr, p_context->p_out_info );
  out_str( p_context->p_out_info, " -> " );
  addr_addr_out( p_dest_addr, p_context->p_out_info );
  if( p_data == p_context->p_stream_list->p_current ) //mark current stream
    out_str( p_context->p_out_info, " <<<=====" );
  out_str( p_context->p_out_info, "\n" );

  //output priority and time data
  out_str( p_context->p_out_info, "    priority: " );
  sprintf( txt, "%u", p_data->priority );
  out_str( p_context->p_out_info, txt );
  out_str( p_context->p_out_info, "\n    timeout: " );
  sprintf( txt, "%u s", p_data->timeout );
  out_str( p_context->p_out_info, txt );
  out_str( p_context->p_out_info, " (" );
  timeout_msec = global_millisecs - p_data->last_frame_msec;
  sprintf( txt, "%u s %u ms", (unsigned short)(timeout_msec / 1000),
                              (unsigned short)(timeout_msec % 1000) );
  out_str( p_context->p_out_info, txt );
  out_str( p_context->p_out_info, ")\n" );

  //continue with enumeration
  return 0;
}

//list streams in a stream list
void stream_list( st_stream_list * p_stream_list, st_out_info * p_out_info )
{
  st_stream_list_context context;

  //enumerate list items
  context.p_stream_list = p_stream_list;
  context.p_out_info = p_out_info;
  addr_list_enum( p_stream_list->p_list, stream_list_enum, (void *)&context );
}

//pass a frame to a stream list
//returns 0 on success, a negative value in case of error
int stream_frame( st_stream_list * p_stream_list, char * p_name, st_frame * p_frame, unsigned short priority, unsigned short timeout )
{
  st_addr * p_src_addr, * p_dest_addr;
  st_stream_data * p_data;
  int ret_val;

  //get source and destination address of frame
  frame_get_addrs( p_frame, &p_src_addr, &p_dest_addr );

  //get item from list
  p_data = addr_list_get( p_stream_list->p_list, p_src_addr, p_dest_addr );
  //not yet in list
  if( p_data == NULL )
  {
    //allocate data item
    p_data = stream_data_new( );
    if( p_data == NULL )
      return -ENOMEM;
    //add stream to list
    ret_val = addr_list_add( p_stream_list->p_list, p_src_addr, p_dest_addr, p_data );
    if( ret_val != 0 )
    {
      stream_data_free( p_data );
      return ret_val;
    }
  }
  
  //copy name
  if( strlen( p_name ) >= list_item_name_size ) //truncate name if too long
    p_name[list_item_name_size-1] = 0;
  strcpy( p_data->name, p_name );

  //save reference to new frame
  if( p_data->p_frame != NULL ) //decrease usage count of old frame
    frame_dec_use( p_data->p_frame );
  p_data->p_frame = p_frame; //save reference to new frame
  frame_inc_use( p_data->p_frame ); //increase usage count of new frame

  //initialize priority and timeout
  p_data->priority = priority;
  p_data->timeout = timeout;
  p_data->timeout_msec = (unsigned long)timeout * 1000;

  //last frame was received now
  p_data->last_frame_msec = global_millisecs;

  //no current stream
  if( p_stream_list->p_current == NULL )
    //make this stream the current one
    p_stream_list->p_current = p_data;
  //stream has got higher priority than the current stream
  else if( p_data->priority > p_stream_list->p_current->priority )
    //make this stream the current one
    p_stream_list->p_current = p_data;

  //if this stream is current stream
  if( p_data == p_stream_list->p_current )
    //send current frame to destinations
    outport_send_frame( p_stream_list->p_outport_data, p_data->p_frame );

  return 0;
}

//do periodic work for a stream list - enumeration deletion function
//item is deleted if return value is != 0
static int stream_tick_enum_del( st_addr * p_src_addr, st_addr * p_dest_addr, void * vp_data, void * vp_stream_list )
{
  st_stream_data * p_data = vp_data;
  st_stream_list * p_stream_list = vp_stream_list;
  unsigned long timeout_msec;

  //check for timeout
  timeout_msec = global_millisecs - p_data->last_frame_msec;
  if( timeout_msec >= p_data->timeout_msec )
  {
    //this is the current item
    if( p_data == p_stream_list->p_current )
      //set current item to none
      p_stream_list->p_current = NULL;
    //delete this item
    return 1;
  }

  //do not delete this item
  return 0;

  //keep compiler happy
  p_src_addr = NULL;
  p_dest_addr = NULL;
}

//do periodic work for a stream list - enumeration function
static int stream_tick_enum( st_addr * p_src_addr, st_addr * p_dest_addr, void * vp_data, void * vp_stream_list )
{
  st_stream_data * p_data = vp_data;
  st_stream_list * p_stream_list = vp_stream_list;

  //no current item
  if( p_stream_list->p_current == NULL )
    //use this item as current one
    p_stream_list->p_current = p_data;
  //item with higher priority
  else if( p_data->priority > p_stream_list->p_current->priority )
    //use this item as current one
    p_stream_list->p_current = p_data;
  
  //continue with enumeration
  return 0;

  //keep compiler happy
  p_src_addr = NULL;
  p_dest_addr = NULL;
}

//do periodic work for a stream list
void stream_tick( st_stream_list * p_stream_list )
{
  //delete some list items by enumeration
  addr_list_enum_del( p_stream_list->p_list, stream_tick_enum_del, (void *)p_stream_list, stream_data_free );

  //current item was removed
  if( p_stream_list->p_current == NULL )
  {
    //find new item
    addr_list_enum( p_stream_list->p_list, stream_tick_enum, (void *)p_stream_list );

    //new current item was found
    if( p_stream_list->p_current != NULL )
      //send current frame of current item to destinations
      outport_send_frame( p_stream_list->p_outport_data, p_stream_list->p_current->p_frame );
  }
}

//send current frame to a destination
void stream_send_current_frame( st_stream_list * p_stream_list, et_proto proto, st_addr * p_addr, st_format * p_format )
{
  int sock;

  //no current frame
  if( p_stream_list->p_current == NULL )
    return;
  
  //send frame
  sock = outport_sock( p_stream_list->p_outport_data );
  proto_send_frame( sock, proto, p_addr, *p_format, p_stream_list->p_current->p_frame );
}
