/* BLINKENLIGHTS PROXY
 * version 0.93 date 2004-04-26
 * Copyright (C) 2003-2004 1stein <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <errno.h>

#include "addr.h"
#include "addr_list.h"
#include "global.h"
#include "list.h"
#include "outport.h"
#include "output.h"
#include "proto.h"
#include "stream.h"
#include "dyndest.h"

//type for data of dynamic destination list items (see header file)
struct s_dyndest_data
{
  char name[list_item_name_size]; //name of this dynamic destination
  et_proto proto; //protocol to use
  st_format format; //format of pictures
  unsigned short timeout; //timeout (in seconds)
  unsigned long timeout_msec; //timeout (in milliseconds)
                              //sort of copy of <timeout>
			      //used to avoid multiplications during ticks
  unsigned long last_request_msec; //msecs when last request was received
};

//type for a dynamic destination list (see header file)
struct s_dyndest_list
{
  st_addr_list * p_list; //the dynamic destination list
  st_outport_data * p_outport_data; //pointer to owning outport
};

//create a new dynamic destination list data item
//returns the pointer to the data item or NULL in case of error
static st_dyndest_data * dyndest_data_new( )
{
  st_dyndest_data * p_data;

  //allocate a new dynamic destination list data item
  p_data = (st_dyndest_data *)malloc( sizeof( st_dyndest_data ) );
  if( p_data == NULL )
    return NULL;

  //initialize data item
  p_data->name[0] = 0; //no name yet
  p_data->proto = proto_none; //no protocol yet
  p_data->format = format_none; //no format yet
  p_data->timeout = 0; //imediate timeout for now
  p_data->timeout_msec = 0;
  p_data->last_request_msec = global_millisecs; //last request was received now

  return p_data;
}

//free dynamic destination list data item
static void dyndest_data_free( void * vp_data )
{
  st_dyndest_data * p_data = vp_data;

  //free data item
  free( p_data );
}

//create a new dynamic destination list
//returns the pointer to the list or NULL in case of error
st_dyndest_list * dyndest_new( st_outport_data * p_outport_data, st_out_info * p_out_info )
{
  st_dyndest_list * p_dyndest_list;

  //create structure
  p_dyndest_list = (st_dyndest_list *)malloc( sizeof( st_dyndest_list ) );
  if( p_dyndest_list == NULL )
  {
    out_str( p_out_info, "  error: could not create new dynamic destination list (out of memory)\n" );
    return NULL;
  }

  //create list
  p_dyndest_list->p_list = addr_list_new( );
  if( p_dyndest_list->p_list == NULL )
  {
    free( p_dyndest_list );
    out_str( p_out_info, "  error: could not create new dynamic destination list\n" );
    return NULL;
  }
  
  //initialize rest of structure
  p_dyndest_list->p_outport_data = p_outport_data; //save pointer to owning outport

  return p_dyndest_list;
}

//free a dynamic destination list
void dyndest_free( st_dyndest_list * p_dyndest_list )
{
  //free list
  addr_list_free( p_dyndest_list->p_list, dyndest_data_free );

  //free structure
  free( p_dyndest_list );
}

//add a dynamic destination
//returns 0 on success, a negative value in case of error
int dyndest_add( st_dyndest_list * p_dyndest_list, st_addr * p_src_addr, st_addr * p_dest_addr, char * p_name, et_proto proto, st_format * p_format, unsigned short timeout )
{
  st_dyndest_data * p_data;
  st_stream_list * p_stream_list;
  int ret_val;

  //get item from list
  p_data = addr_list_get( p_dyndest_list->p_list, p_src_addr, p_dest_addr );
  //not yet in list
  if( p_data == NULL )
  {
    //allocate data item
    p_data = dyndest_data_new( );
    if( p_data == NULL )
      return -ENOMEM;
    //add dynamic destination to list
    ret_val = addr_list_add( p_dyndest_list->p_list, p_src_addr, p_dest_addr, p_data );
    if( ret_val != 0 )
    {
      dyndest_data_free( p_data );
      return ret_val;
    }
  }

  //copy name
  if( strlen( p_name ) >= list_item_name_size ) //truncate name if too long
    p_name[list_item_name_size-1] = 0;
  strcpy( p_data->name, p_name );

  //save protocol and format
  p_data->proto = proto;
  p_data->format = *p_format;

  //initialize timeout
  p_data->timeout = timeout;
  p_data->timeout_msec = (unsigned long)timeout * 1000;

  //last request was received now
  p_data->last_request_msec = global_millisecs;

  //send current frame to new destination
  p_stream_list = outport_stream_list( p_dyndest_list->p_outport_data );
  stream_send_current_frame( p_stream_list, p_data->proto, p_dest_addr, &p_data->format );

  return 0;
}

//remove a dynamic destination
//returns 0 on success, a negative value in case of error
int dyndest_del( st_dyndest_list * p_dyndest_list, st_addr * p_src_addr, st_addr * p_dest_addr )
{
  //delete item from list
  return addr_list_del( p_dyndest_list->p_list, p_src_addr, p_dest_addr, dyndest_data_free );
}

//list dynamic destinations - enumeration function
static int dyndest_list_enum( st_addr * p_src_addr, st_addr * p_dest_addr, void * vp_data, void * vp_out_info )
{
  st_dyndest_data * p_data = vp_data;
  st_out_info * p_out_info = vp_out_info;
  unsigned long timeout_msec;
  char txt[32];

  //output name, source address and destination address
  out_str( p_out_info, "  " );
  out_str( p_out_info, p_data->name );
  out_str( p_out_info, ": " );
  addr_addr_out( p_src_addr, p_out_info );
  out_str( p_out_info, " -> " );
  addr_addr_out( p_dest_addr, p_out_info );
  out_str( p_out_info, "\n" );

  //output priority and time data
  out_str( p_out_info, "    timeout: " );
  sprintf( txt, "%u s", p_data->timeout );
  out_str( p_out_info, txt );
  out_str( p_out_info, " (" );
  timeout_msec = global_millisecs - p_data->last_request_msec;
  sprintf( txt, "%u s %u ms", (unsigned short)(timeout_msec / 1000),
                              (unsigned short)(timeout_msec % 1000) );
  out_str( p_out_info, txt );
  out_str( p_out_info, ")\n" );

  //continue with enumeration
  return 0;
}

//list dynamic destinations in a dynamic destination list
void dyndest_list( st_dyndest_list * p_dyndest_list, st_out_info * p_out_info )
{
  //enumerate list items
  addr_list_enum( p_dyndest_list->p_list, dyndest_list_enum, (void *)p_out_info );
}

//do periodic work for a dynamic destination list - enumeration deletion function
//item is deleted if return value is != 0
static int dyndest_tick_enum_del( st_addr * p_src_addr, st_addr * p_dest_addr, void * vp_data, void * vp_unused )
{
  st_dyndest_data * p_data = vp_data;
  unsigned long timeout_msec;

  //check for timeout
  timeout_msec = global_millisecs - p_data->last_request_msec;
  if( timeout_msec >= p_data->timeout_msec )
    //delete this item
    return 1;

  //do not delete this item
  return 0;

  //keep compiler happy
  p_src_addr = NULL;
  p_dest_addr = NULL;
  vp_unused = NULL;
}

//do periodic work for a dynamic destination list
void dyndest_tick( st_dyndest_list * p_dyndest_list )
{
  //delete some list items by enumeration
  addr_list_enum_del( p_dyndest_list->p_list, dyndest_tick_enum_del, NULL, dyndest_data_free );
}

//send a frame to dynamic destinations - type for enumeration context
typedef struct s_dyndest_send_frame_context
{
  st_frame * p_frame;
  int sock;
} st_dyndest_send_frame_context;

//send a frame to dynamic destinations - enumeration function
static int dyndest_send_frame_enum( st_addr * p_src_addr, st_addr * p_dest_addr, void * vp_data, void * vp_context )
{
  st_dyndest_data * p_data = vp_data;
  st_dyndest_send_frame_context * p_context = vp_context;

  //send frame
  proto_send_frame( p_context->sock, p_data->proto, p_dest_addr, p_data->format, p_context->p_frame );

  //continue with enumeration
  return 0;

  //keep compiler happy
  p_src_addr = NULL;
}

//send a frame to dynamic destinations
void dyndest_send_frame( st_dyndest_list * p_dyndest_list, st_frame * p_frame, int sock )
{
  st_dyndest_send_frame_context context;

  //enumerate list items
  context.p_frame = p_frame;
  context.sock = sock;
  addr_list_enum( p_dyndest_list->p_list, dyndest_send_frame_enum, (void *)&context );
}
