/* BLINKENLIGHTS PROXY
 * version 0.92 date 2004-04-07
 * Copyright (C) 2003 1stein <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>

#include "addr.h"
#include "output.h"
#include "proto.h"



//protocol magics
#define BLP_MAGIC "\xDE""\xAD""\xBE""\xEF"
#define BLP_MAGIC_LEN 4
#define EBLP_MAGIC "\xFE""\xED""\xBE""\xEF"
#define EBLP_MAGIC_LEN 4
#define MCUF_MAGIC "\x23""\x54""\x26""\x66"
#define MCUF_MAGIC_LEN 4

//protocol headers
typedef struct s_proto_blp_hdr
{
  uint32_t magic;
  uint32_t frame_no;
  uint16_t width;
  uint16_t height;
} st_proto_blp_hdr;
#define proto_blp_magic 0xDEADBEEF
typedef struct s_proto_eblp_hdr
{
  uint32_t magic;
  uint32_t frame_no;
  uint16_t width;
  uint16_t height;
} st_proto_eblp_hdr;
#define proto_eblp_magic 0xFEEDBEEF
typedef struct s_proto_mcuf_hdr
{
  uint32_t magic;
  uint16_t height;
  uint16_t width;
  uint16_t channels;
  uint16_t maxval;
} st_proto_mcuf_hdr;
#define proto_mcuf_magic 0x23542666

//refresh messages
#define BLP_REFRESH "\xDE""\xAA""\xBF""\xCD""REFRESH"
#define BLP_REFRESH_LEN 11
#define EBLP_REFRESH "\xDE""\xAA""\xBF""\xCD""REFRESH256"
#define EBLP_REFRESH_LEN 14
#define MCUF_REFRESH "\x42""\x42""\x42""\x42""\0\0\0\0\0\0\0\0"
#define MCUF_REFRESH_LEN 12

//close messages
#define BLP_CLOSE "\xDE""\xAA""\xBF""\xCD""CLOSE"
#define BLP_CLOSE_LEN 9
#define EBLP_CLOSE "\xDE""\xAA""\xBF""\xCD""CLOSE256"
#define EBLP_CLOSE_LEN 12
#define MCUF_CLOSE "\x42""\x42""\x42""\x43""\0\0\0\0\0\0\0\0"
#define MCUF_CLOSE_LEN 12



//read protocol from string
//<proto_any_ok> != 0 allows return value <proto_any>
//returns protocol on success, proto_none on error
et_proto proto_str2proto( char * p_str, int proto_any_ok, st_out_info * p_out_info )
{
  //BLP
  if( strcasecmp( p_str, "blp" ) == 0 )
    return proto_blp;
  //EBLP
  else if( strcasecmp( p_str, "eblp" ) == 0 )
    return proto_eblp;
  //MCUF
  else if( strcasecmp( p_str, "mcuf" ) == 0 )
    return proto_mcuf;
  //any protocol
  else if( strcasecmp( p_str, "any" ) == 0 && proto_any_ok )
    return proto_any;
  //unknown protocol
  else
  {
    out_str( p_out_info, "  error: unknown protocol \"" );
    out_str( p_out_info, p_str );
    out_str( p_out_info, "\" (try \"BLP\", \"EBLP\" or \"MCUF\")\n" );
    return proto_none;
  }
}

//get string from protocol
//returns 0 on success, a negative value on error
int proto_proto2str( et_proto proto, char * p_str, unsigned int str_buf_size, st_out_info * p_out_info )
{
  //switch by protocol
  switch( proto )
  {
    //BLP
    case proto_blp:
      //return string
      if( str_buf_size < 4 )
      {
        out_str( p_out_info, "  error: cannot convert protocol to string (buffer too small)\n" );
        return -ENOMEM;
      }
      strcpy( p_str, "BLP" );
      return 0;
    //EBLP
    case proto_eblp:
      //return string
      if( str_buf_size < 5 )
      {
        out_str( p_out_info, "  error: cannot convert protocol to string (buffer too small)\n" );
        return -ENOMEM;
      }
      strcpy( p_str, "EBLP" );
      return 0;
    //MCUF
    case proto_mcuf:
      //return string
      if( str_buf_size < 5 )
      {
        out_str( p_out_info, "  error: cannot convert protocol to string (buffer too small)\n" );
        return -ENOMEM;
      }
      strcpy( p_str, "MCUF" );
      return 0;
    //any protocol
    case proto_any:
      //return string
      if( str_buf_size < 4 )
      {
        out_str( p_out_info, "  error: cannot convert protocol to string (buffer too small)\n" );
        return -ENOMEM;
      }
      strcpy( p_str, "any" );
      return 0;
    //unknown protocol
    case proto_none:
    default:
      out_str( p_out_info, "  error: unknown protocol\n" );
      return -EINVAL;
  } //switch( proto )
}

//output protocol
void proto_proto_out( et_proto proto, st_out_info * p_out_info )
{
  //switch by protocol
  switch( proto )
  {
    //BLP
    case proto_blp:
      out_str( p_out_info, "BLP" );
      break;
    //EBLP
    case proto_eblp:
      out_str( p_out_info, "EBLP" );
      break;
    //MCUF
    case proto_mcuf:
      out_str( p_out_info, "MCUF" );
      break;
    //any protocol
    case proto_any:
      out_str( p_out_info, "any" );
      break;
    //unknown protocol
    case proto_none:
    default:
      out_str( p_out_info, "<unknown protocol>" );
      break;
  } //switch( proto )
}



//read format from string
//format of <format> ist [<width>x<height>][-<channels>][/<colors>]
//returns 0 on success, a negative value on error
int format_str2format( char * p_str, st_format * p_format, st_out_info * p_out_info )
{
  char * p_chr, * p_width, * p_height, * p_channels, * p_colors;
  unsigned long value;

  //get colors
  p_chr = strchr( p_str, '/' );
  p_colors = "";
  if( p_chr != NULL )
  {
    *p_chr = 0;
    p_colors = p_chr + 1;
  }

  //get channels
  p_chr = strchr( p_str, '-' );
  p_channels = "";
  if( p_chr != NULL )
  {
    *p_chr = 0;
    p_channels = p_chr + 1;
  }

  //get width and height
  p_width = "";
  p_height = "";
  if( *p_str != 0 )
  {
    p_chr = strchr( p_str, 'x' );
    if( p_chr == NULL )
    {
      out_str( p_out_info, "  error: invalid dimensions \"" );
      out_str( p_out_info, p_str );
      out_str( p_out_info, "\"\n" );
      return -EINVAL;
    }
    *p_chr = 0;
    p_width = p_str;
    p_height = p_chr + 1;
  }

  //parse height
  value = strtoul( p_height, &p_chr, 0 );
  if( *p_height == 0 )
    p_format->height = 0;
  else
  {
    if( *p_chr != 0 || value == 0 || value > format_height_max )
    {
      out_str( p_out_info, "  error: invalid height \"" );
      out_str( p_out_info, p_height );
      out_str( p_out_info, "\"\n" );
      return -EINVAL;
    }
    p_format->height = (unsigned short)value;
  }

  //parse width
  value = strtoul( p_width, &p_chr, 0 );
  if( *p_width == 0 )
    p_format->width = 0;
  else
  {
    if( *p_chr != 0 || value == 0 || value > format_width_max )
    {
      out_str( p_out_info, "  error: invalid width \"" );
      out_str( p_out_info, p_width );
      out_str( p_out_info, "\"\n" );
      return -EINVAL;
    }
    p_format->width = (unsigned short)value;
  }

  //parse channels
  value = strtoul( p_channels, &p_chr, 0 );
  if( *p_channels == 0 )
    p_format->channels = 0;
  else
  {
    if( *p_chr != 0 || value == 0 || value > format_channels_max )
    {
      out_str( p_out_info, "  error: invalid number of channels \"" );
      out_str( p_out_info, p_channels );
      out_str( p_out_info, "\"\n" );
      return -EINVAL;
    }
    p_format->channels = (unsigned short)value;
  }

  //parse colors
  value = strtoul( p_colors, &p_chr, 0 );
  if( *p_colors == 0 )
    p_format->colors = 0;
  else
  {
    if( *p_chr != 0 || value <= 1 || value > format_colors_max )
    {
      out_str( p_out_info, "  error: invalid number of colors \"" );
      out_str( p_out_info, p_colors );
      out_str( p_out_info, "\"\n" );
      return -EINVAL;
    }
    p_format->colors = (unsigned short)value;
  }

  return 0;
}

//get string from format
//format of <format> ist [<width>x<height>][-<channels>][/<colors>]
//returns 0 on success, a negative value on error
int format_format2str( st_format * p_format, char * p_str, unsigned int str_buf_size, st_out_info * p_out_info )
{
  //check size
  if( str_buf_size < 11 + 2 + 4 + 1 )
  {
    out_str( p_out_info, "  error: cannot convert format to string (buffer too small)\n" );
    return -ENOMEM;
  }

  //write dimensions
  if( p_format->height > 0 && p_format->height <= format_height_max && p_format->width > 0 && p_format->width <= format_width_max )
  {
    sprintf( p_str, "%ux%u", p_format->width, p_format->height );
    p_str += strlen( p_str );
  }

  //write number of channels
  if( p_format->channels > 0 && p_format->channels <= format_channels_max )
  {
    sprintf( p_str, "-%u", p_format->channels );
    p_str += strlen( p_str );
  }

  //write number of colors
  if( p_format->colors > 1 && p_format->colors <= format_colors_max )
  {
    sprintf( p_str, "/%u", p_format->colors );
    p_str += strlen( p_str );
  }

  return 0;
}

//output format
//format of <format> ist [<width>x<height>][-<channels>][/<colors>]
void format_format_out( st_format * p_format, st_out_info * p_out_info )
{
  char txt[12];

  //output dimensions
  if( p_format->height > 0 && p_format->height <= format_height_max && p_format->width > 0 && p_format->width <= format_width_max )
  {
    sprintf( txt, "%ux%u", p_format->width, p_format->height );
    out_str( p_out_info, txt );
  }

  //output number of channels
  if( p_format->channels > 0 && p_format->channels <= format_channels_max )
  {
    sprintf( txt, "-%u", p_format->channels );
    out_str( p_out_info, txt );
  }

  //output number of colors
  if( p_format->colors > 1 && p_format->colors <= format_colors_max )
  {
    sprintf( txt, "/%u", p_format->colors );
    out_str( p_out_info, txt );
  }
}

//check if a certain format matches a general format
//retuns 1 on match, 0 on mismatch
int format_match( st_format * p_certain_format, st_format * p_general_format )
{
  if( p_certain_format->height != p_general_format->height && p_general_format->height != 0 )
    return 0;
  if( p_certain_format->width != p_general_format->width && p_general_format->width != 0 )
    return 0;
  if( p_certain_format->channels != p_general_format->channels && p_general_format->channels != 0 )
    return 0;
  if( p_certain_format->colors != p_general_format->colors && p_general_format->colors != 0 )
    return 0;
  return 1;
}



//type for a frame (see header file)
struct s_frame
{
  unsigned int use_cnt; //number of references to this frame
  st_addr src_addr, dest_addr; //source and destination address of the packet the frame was in
  char * p_msg; //pointer to the malloc-ed memory block with the message
  unsigned int msg_len; //length of the message
  et_proto proto; //the protocol of this frame
  st_format format; //the format of this frame
  char * p_msg_data; //pointer into the message block to the starting point of the real data
};

//parse message and generate a frame from it
//p_msg must point to a malloc-ed memory block with a message
//p_msg belongs to the frame after successful execution of this procedure
//returns a pointer to frame with initial use count of 1
//returns NULL in case of error
st_frame * frame_parse( st_addr * p_src_addr, st_addr * p_dest_addr, char * p_msg, unsigned int msg_len )
{
  et_proto proto;
  st_format format;
  char * p_msg_data;
  st_frame * p_frame;

  //try to identify the protocol by the magic
  
  //BLP
  if( msg_len >= sizeof( st_proto_blp_hdr ) && ntohl( ((st_proto_blp_hdr *)p_msg)->magic ) == proto_blp_magic )
  {
    proto = proto_blp;

    //get, check and store format

    //height
    format.height = ntohs( ((st_proto_blp_hdr *)p_msg)->height );
    if( format.height == 0 || format.height > format_height_max )
      return NULL;
    //width
    format.width = ntohs( ((st_proto_blp_hdr *)p_msg)->width );
    if( format.width == 0 || format.width > format_width_max )
      return NULL;
    //channels: 1
    format.channels = 1;
    //colors: 2
    format.colors = 2;

    //check size of message
    if( msg_len < sizeof( st_proto_blp_hdr ) + format.height * format.width * format.channels )
      return NULL;

    //set pointer to real data
    p_msg_data = p_msg + sizeof( st_proto_blp_hdr );
  }
  
  //EBLP
  else if( msg_len >= sizeof( st_proto_eblp_hdr ) && ntohl( ((st_proto_eblp_hdr *)p_msg)->magic ) == proto_eblp_magic )
  {
    proto = proto_eblp;

    //get, check and store format

    //height
    format.height = ntohs( ((st_proto_blp_hdr *)p_msg)->height );
    if( format.height == 0 || format.height > format_height_max )
      return NULL;
    //width
    format.width = ntohs( ((st_proto_blp_hdr *)p_msg)->width );
    if( format.width == 0 || format.width > format_width_max )
      return NULL;
    //channels: 1
    format.channels = 1;
    //colors: 256
    format.colors = 256;

    //check size of message
    if( msg_len < sizeof( st_proto_eblp_hdr ) + format.height * format.width * format.channels )
      return NULL;

    //set pointer to real data
    p_msg_data = p_msg + sizeof( st_proto_eblp_hdr );
  }
  
  //MCUF
  else if( msg_len >= sizeof( st_proto_mcuf_hdr ) && ntohl( ((st_proto_mcuf_hdr *)p_msg)->magic ) == proto_mcuf_magic )
  {
    proto = proto_mcuf;

    //get, check and store format

    //height
    format.height = ntohs( ((st_proto_mcuf_hdr *)p_msg)->height );
    if( format.height == 0 || format.height > format_height_max )
      return NULL;
    //width
    format.width = ntohs( ((st_proto_mcuf_hdr *)p_msg)->width );
    if( format.width == 0 || format.width > format_width_max )
      return NULL;
    //channels
    format.channels = ntohs( ((st_proto_mcuf_hdr *)p_msg)->channels );
    if( format.channels == 0 || format.channels > format_channels_max )
      return NULL;
    //colors
    format.colors = ntohs( ((st_proto_mcuf_hdr *)p_msg)->maxval ) + 1;
    if( format.colors <= 1 || format.colors > format_colors_max )
      return NULL;

    //check size of message
    if( msg_len < sizeof( st_proto_mcuf_hdr ) + format.height * format.width * format.channels )
      return NULL;

    //set pointer to real data
    p_msg_data = p_msg + sizeof( st_proto_blp_hdr );
  }
  
  //unknown protocol
  else
    return NULL;
  
  //allocate a frame structure
  p_frame = (st_frame *)malloc( sizeof( st_frame ) );
  if( p_frame == NULL )
    return NULL;

  //set up frame structure
  p_frame->use_cnt = 1;
  p_frame->src_addr = *p_src_addr;
  p_frame->dest_addr = *p_dest_addr;
  p_frame->p_msg = p_msg;
  p_frame->msg_len = msg_len;
  p_frame->proto = proto;
  p_frame->format = format;
  p_frame->p_msg_data = p_msg_data;

  return p_frame;
}

//get pointers to the addresses of the frame
void frame_get_addrs( st_frame * p_frame, st_addr ** pp_src_addr, st_addr ** pp_dest_addr )
{
  *pp_src_addr = &p_frame->src_addr;
  *pp_dest_addr = &p_frame->dest_addr;
}

//increment use count for a frame
void frame_inc_use( st_frame * p_frame )
{
  //increment use count
  p_frame->use_cnt++;
}

//decrement use count for a frame
void frame_dec_use( st_frame * p_frame )
{
  //decrement use count
  p_frame->use_cnt--;

  //use count reached zero
  if( p_frame->use_cnt == 0 )
  {
    //free memory block with message
    free( p_frame->p_msg );

    //free frame structure
    free( p_frame );
  }
}

//check if a frame matches an address range, a protocol and a format
//retuns 1 on match, 0 on mismatch
int frame_match( st_frame * p_frame, st_addr_range * p_range, et_proto proto, st_format * p_format )
{
  //check if address matches
  if( ! addr_match( &p_frame->src_addr, p_range ) )
    return 0;

  //check if protocol matches
  if( p_frame->proto != proto && proto != proto_any )
    return 0;

  //check if format matches
  if( ! format_match( &p_frame->format, p_format ) )
    return 0;

  return 1;
}



//convert frame to new format
//destination data should be initialized to zero because unused areas are not set to zero here
static void frame_convert( st_format * p_src_format, unsigned char * p_src_data, st_format * p_dest_format, unsigned char * p_dest_data )
{
  unsigned short src_skip_x, dest_skip_x, src_ofs_x, src_ofs_y, dest_ofs_x, dest_ofs_y, size_x, size_y, x, y, c;
  unsigned long src_skip, dest_skip, src_i, dest_i;
  unsigned short val, src_maxval, dest_maxval, src_div;

  //get y values
  if( p_src_format->height < p_dest_format->height )
  {
    src_ofs_y = 0;
    dest_ofs_y = (p_dest_format->height - p_src_format->height) / 2;
    size_y = p_src_format->height;
  }
  else
  {
    src_ofs_y = (p_src_format->height - p_dest_format->height) / 2;
    dest_ofs_y = 0;
    size_y = p_dest_format->height;
  }
  //get x values
  if( p_src_format->width < p_dest_format->width )
  {
    src_skip_x = 0;
    dest_skip_x = p_dest_format->width - p_src_format->width;
    src_ofs_x = 0;
    dest_ofs_x = dest_skip_x / 2;
    size_x = p_src_format->width;
  }
  else
  {
    src_skip_x = p_src_format->width - p_dest_format->width;
    dest_skip_x = 0;
    src_ofs_x = src_skip_x / 2;
    dest_ofs_x = 0;
    size_x = p_dest_format->width;
  }

  //convert
  src_skip = src_skip_x * p_src_format->channels;
  dest_skip = dest_skip_x * p_dest_format->channels;
  src_i = (src_ofs_y * p_src_format->width + src_ofs_x) * p_src_format->channels;
  dest_i = (dest_ofs_y * p_dest_format->width + dest_ofs_x) * p_dest_format->channels;
  //number of channels is equal
  if( p_src_format->channels == p_dest_format->channels )
  {
    //number of colors is equal
    if( p_src_format->colors == p_dest_format->colors )
    {
      for( y = 0; y < size_y; y++ )
      {
        for( x = 0; x < size_x; x++ )
          for( c = 0; c < p_src_format->channels; c++ )
            p_dest_data[dest_i++] = p_src_data[src_i++];
        src_i += src_skip;
        dest_i += dest_skip;
      }
    }
    //number of colors is _not_ equal
    else
    {
      src_maxval = p_src_format->colors - 1;
      dest_maxval = p_dest_format->colors - 1;
      for( y = 0; y < size_y; y++ )
      {
        for( x = 0; x < size_x; x++ )
        {
          for( c = 0; c < p_src_format->channels; c++ )
          {
            val = p_src_data[src_i++];
            val = (val * dest_maxval) / src_maxval;
            p_dest_data[dest_i++] = (unsigned char)val;
          }
        }
        src_i += src_skip;
        dest_i += dest_skip;
      }
    }
  }
  //number of channels is _not_ equal
  else
  {
    //number of colors is equal
    if( p_src_format->colors == p_dest_format->colors )
    {
      for( y = 0; y < size_y; y++ )
      {
        for( x = 0; x < size_x; x++ )
        {
          val = 0;
          for( c = 0; c < p_src_format->channels; c++ )
            val += p_src_data[src_i++];
          val /= p_src_format->channels;
          for( c = 0; c < p_dest_format->channels; c++ )
            p_dest_data[dest_i++] = (unsigned char)val;
        }
        src_i += src_skip;
        dest_i += dest_skip;
      }
    }
    //number of colors is _not_ equal
    else
    {
      src_maxval = p_src_format->colors - 1;
      dest_maxval = p_dest_format->colors - 1;
      src_div = src_maxval * p_src_format->channels;
      for( y = 0; y < size_y; y++ )
      {
        for( x = 0; x < size_x; x++ )
        {
          val = 0;
          for( c = 0; c < p_src_format->channels; c++ )
            val += p_src_data[src_i++];
          val = (val * dest_maxval) / src_div;
          for( c = 0; c < p_dest_format->channels; c++ )
            p_dest_data[dest_i++] = (unsigned char)val;
        }
        src_i += src_skip;
        dest_i += dest_skip;
      }
    }
  }
}



//send dynamic stream refresh
void proto_send_refresh( int sock, et_proto proto, st_addr * p_addr )
{
  //switch by protocol
  switch( proto )
  {
    //BLP
    case proto_blp:
      sendto( sock, BLP_REFRESH, BLP_REFRESH_LEN, 0, &p_addr->sa.sa, p_addr->sa_len );
      break;
    //EBLP
    case proto_eblp:
      sendto( sock, EBLP_REFRESH, EBLP_REFRESH_LEN, 0, &p_addr->sa.sa, p_addr->sa_len );
      break;
    //MCUF
    case proto_mcuf:
      sendto( sock, MCUF_REFRESH, MCUF_REFRESH_LEN, 0, &p_addr->sa.sa, p_addr->sa_len );
      break;
    //unknown protocol
    case proto_none:
    default:
      break;
  } //switch( proto )
}

//send dynamic stream close
void proto_send_close( int sock, et_proto proto, st_addr * p_addr )
{
  //switch by protocol
  switch( proto )
  {
    //BLP
    case proto_blp:
      sendto( sock, BLP_CLOSE, BLP_CLOSE_LEN, 0, &p_addr->sa.sa, p_addr->sa_len );
      break;
    //EBLP
    case proto_eblp:
      sendto( sock, EBLP_CLOSE, EBLP_CLOSE_LEN, 0, &p_addr->sa.sa, p_addr->sa_len );
      break;
    //MCUF
    case proto_mcuf:
      sendto( sock, MCUF_CLOSE, MCUF_CLOSE_LEN, 0, &p_addr->sa.sa, p_addr->sa_len );
      break;
    //unknown protocol
    case proto_none:
    default:
  } //switch( proto )
}

//send a frame
void proto_send_frame( int sock, et_proto proto, st_addr * p_addr, st_format format, st_frame * p_frame )
{
  char * p_msg, * p_msg_data;
  int msg_len, msg_data_len;
  st_proto_blp_hdr * p_blp_hdr;
  st_proto_eblp_hdr * p_eblp_hdr;
  st_proto_mcuf_hdr * p_mcuf_hdr;

  //fill general format parts with certain data
  if( format.height == 0 )
    format.height = p_frame->format.height;
  if( format.width == 0 )
    format.width = p_frame->format.width;
  if( format.channels == 0 )
    format.channels = p_frame->format.channels;
  if( format.colors == 0 )
    format.colors = p_frame->format.colors;

  //adapt format
  switch( proto ) //switch by protocol
  {
    //BLP
    case proto_blp:
      format.channels = 1;
      format.colors = 2;
      break;
    //EBLP
    case proto_eblp:
      //adapt format
      format.channels = 1;
      format.colors = 256;
      break;
    //MCUF
    case proto_mcuf:
      break;
    //unknown protocol
    case proto_none:
    default:
  } //switch( proto )

  //formats are exactly the same
  if( p_frame->format.height == format.height
   && p_frame->format.width == format.width
   && p_frame->format.channels == format.channels
   && p_frame->format.colors == format.colors )
  {
    //send original packet
    sendto( sock, p_frame->p_msg, p_frame->msg_len, 0, &p_addr->sa.sa, p_addr->sa_len );
    //done
    return;
  }

  //switch by protocol
  switch( proto )
  {
    //BLP
    case proto_blp:
      //allocate buffer for message
      msg_data_len = format.height * format.width;
      msg_len = sizeof( st_proto_blp_hdr ) + msg_data_len;
      p_msg = (char * )alloca( msg_len );
      if( p_msg == NULL )
        break;
      //initialize header
      p_blp_hdr = (st_proto_blp_hdr *)p_msg;
      p_blp_hdr->magic = htonl( proto_blp_magic );
      p_blp_hdr->frame_no = htonl( 0 );
      p_blp_hdr->width = htons( format.width );
      p_blp_hdr->height = htons( format.height );
      //get pointer to message data
      p_msg_data = p_msg + sizeof( st_proto_blp_hdr );
      break;

    //EBLP
    case proto_eblp:
      //allocate buffer for message
      msg_data_len = format.height * format.width;
      msg_len = sizeof( st_proto_eblp_hdr ) + msg_data_len;
      p_msg = (char * )alloca( msg_len );
      if( p_msg == NULL )
        break;
      //initialize header
      p_eblp_hdr = (st_proto_eblp_hdr *)p_msg;
      p_eblp_hdr->magic = htonl( proto_eblp_magic );
      p_eblp_hdr->frame_no = htonl( 0 );
      p_eblp_hdr->width = htons( format.width );
      p_eblp_hdr->height = htons( format.height );
      //get pointer to message data
      p_msg_data = p_msg + sizeof( st_proto_eblp_hdr );
      break;

    //MCUF
    case proto_mcuf:
      //allocate buffer for message
      msg_data_len = format.height * format.width * format.channels;
      msg_len = sizeof( st_proto_mcuf_hdr ) + msg_data_len;
      p_msg = (char * )alloca( msg_len );
      if( p_msg == NULL )
        break;
      //initialize header
      p_mcuf_hdr = (st_proto_mcuf_hdr *)p_msg;
      p_mcuf_hdr->magic = htonl( proto_mcuf_magic );
      p_mcuf_hdr->height = htons( format.height );
      p_mcuf_hdr->width = htons( format.width );
      p_mcuf_hdr->channels = htons( format.channels );
      p_mcuf_hdr->maxval = htons( format.colors - 1 );
      //get pointer to message data
      p_msg_data = p_msg + sizeof( st_proto_mcuf_hdr );
      break;

    //unknown protocol
    case proto_none:
    default:
      //do not send frame
      return;
  } //switch( proto )

  //initialize message data with 0
  memset( p_msg_data, 0, msg_data_len );

  //convert frame to new format
  frame_convert( &p_frame->format, p_frame->p_msg_data, &format, p_msg_data );

  //send new packet
  sendto( sock, p_msg, msg_len, 0, &p_addr->sa.sa, p_addr->sa_len );
}

//parse a request
//returns protocol or proto_none on error
//*p_enable is set to 1 if sending frames should be enabled
et_proto proto_parse_request( char * p_msg, int len, int * p_enable )
{
  //refresh messages
  if( len == BLP_REFRESH_LEN && memcmp( p_msg, BLP_REFRESH, len ) == 0 )
  {
    *p_enable = 1;
    return proto_blp;
  }
  if( len == EBLP_REFRESH_LEN && memcmp( p_msg, EBLP_REFRESH, len ) == 0 )
  {
    *p_enable = 1;
    return proto_eblp;
  }
  if( len == MCUF_REFRESH_LEN && memcmp( p_msg, MCUF_REFRESH, len ) == 0 )
  {
    *p_enable = 1;
    return proto_mcuf;
  }

  //close messages
  if( len == BLP_CLOSE_LEN && memcmp( p_msg, BLP_CLOSE, len ) == 0 )
  {
    *p_enable = 0;
    return proto_blp;
  }
  if( len == EBLP_CLOSE_LEN && memcmp( p_msg, EBLP_CLOSE, len ) == 0 )
  {
    *p_enable = 0;
    return proto_eblp;
  }
  if( len == MCUF_CLOSE_LEN && memcmp( p_msg, MCUF_CLOSE, len ) == 0 )
  {
    *p_enable = 0;
    return proto_mcuf;
  }

  //unknown request
  return proto_none;
}

//check if request matches an address range and a protocol
//retuns 1 on match, 0 on mismatch
int proto_request_match( st_addr * p_addr, et_proto proto, st_addr_range * p_range, et_proto proto_range )
{
  //check if address matches
  if( ! addr_match( p_addr, p_range ) )
    return 0;

  //check if protocol matches
  if( proto != proto_range && proto_range != proto_any )
    return 0;

  return 1;
}
