/* BLINKENLIGHTS PROXY
 * version 0.92 date 2004-04-07
 * Copyright (C) 2003 1stein <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 */

#include <malloc.h>
#include <errno.h>

#include "list.h"

//type for a list item
typedef struct s_list_item
{
  char name[list_item_name_size]; //name of this item
  void * p_data; //data of this item
} st_list_item;

//type for a list (see header file)
struct s_list
{
  unsigned int len; //current length of the list (measured in entries)
  unsigned int size; //current size of the list (measured in entries)
  unsigned int size_step; //current size step of the list (measured in entries)
  st_list_item * * p_list; //pointer to array containing list
};

//get position for/of entry
static void list_get_pos( st_list_item * * p_list, unsigned int len, char * name, int * p_start, int * p_end )
{
  int i;
  int cmp;

  //empty list
  if( len == 0 )
  {
    //insert at position 0
    *p_start = -1;
    *p_end = 0;
    return;
  }

  //start with entire range
  *p_start = 0;
  *p_end = (int)len - 1;

  //do a binary search
  for( ; ; )
  {
    //get middle of range
    i = (*p_start + *p_end) / 2;
    //compare
    cmp = strcmp( name, p_list[i]->name );
    //searched item is above middle
    if( cmp < 0 )
    {
      //not found
      if( i <= *p_start )
      {
        //insert at position i
        *p_start = i - 1;
        *p_end = i;
        break;
      }
      //search above middle
      *p_end = i - 1;
    }
    //searched item is below middle
    else if( cmp > 0 )
    {
      //not found
      if( i >= *p_end )
      {
        //insert at position i + 1
        *p_start = i;
        *p_end = i + 1;
        break;
      }
      //search in below middle
      *p_start = i + 1;
    }
    //found
    else
    {
      //found at position i
      *p_start = i;
      *p_end = i;
      break;
    }
  }
}

//create a new list
//returns NULL in case of error
st_list * list_new( )
{
  st_list * p_list;

  //allocate memeory for list head
  p_list = (st_list *)malloc( sizeof( st_list ) );
  if( p_list == NULL )
    return NULL;

  //initialize list
  p_list->len = 0; //list is empty
  p_list->size = 16; //start with space for some entries
  p_list->size_step = 4; //increase size in steps of quarter to almost half size
  p_list->p_list = (st_list_item * *)malloc( p_list->size * sizeof( st_list_item * ) );
  if( p_list->p_list == NULL )
  {
    free( p_list );
    return NULL;
  }

  return p_list;
}

//delete a list (including all entries)
//data_free_func is called for the data of every list item (if not NULL)
void list_free( st_list * p_list, void (*data_free_func)( void * ) )
{
  unsigned int i;

  //free list entries
  for( i = 0; i < p_list->len; i++ )
  {
    //free data
    if( p_list->p_list[i]->p_data != NULL && data_free_func != NULL )
      data_free_func( p_list->p_list[i]->p_data );
    //free list entry
    free( p_list->p_list[i] );
  }

  //free array and list head
  free( p_list->p_list );
  free( p_list );
}

//add an item to a list
//returns 0 on success, a negative value on error
int list_add( st_list * p_list, char * name, void * p_data )
{
  int start, end, i;
  st_list_item * p_item, * * p_new_list;

  //truncate name
  if( strlen( name ) >= list_item_name_size )
    name[list_item_name_size-1] = 0;

  //get position
  list_get_pos( p_list->p_list, p_list->len, name, &start, &end );
  //already in list
  if( start == end )
    return -EEXIST;

  //allocate list item
  p_item = (st_list_item *)malloc( sizeof( st_list_item ) );
  if( p_item == NULL )
    return -ENOMEM;

  //initialize item
  strcpy( p_item->name, name ); //name is already truncated to maximum length
  p_item->p_data = p_data;

  //increase length of list
  p_list->len++;

  //must size be increased?
  if( p_list->len > p_list->size )
  {
    //must size step be increased?
    if( p_list->size >= p_list->size_step << 2 )
      p_list->size_step <<= 1;
    //increase size
    p_list->size += p_list->size_step;
    //reallocate list
    p_new_list = (st_list_item * *)realloc( p_list->p_list, p_list->size * sizeof( st_list_item * ) );
    if( p_new_list == NULL )
    {
      free( p_item );
      return -ENOMEM;
    }
    p_list->p_list = p_new_list;
  }

  //put new item into list
  for( i = p_list->len; i > end; i-- )
    p_list->p_list[i] = p_list->p_list[i-1];
  p_list->p_list[end] = p_item;

  return 0;
}

//delete an item from a list
//data_free_func is called for the data of deleted list item (if not NULL)
//returns 0 on success, a negative value on error
int list_del( st_list * p_list, char * name, void (*data_free_func)( void * ) )
{
  int start, end, i;
  st_list_item * * p_new_list;

  //truncate name
  if( strlen( name ) >= list_item_name_size )
    name[list_item_name_size-1] = 0;

  //get position
  list_get_pos( p_list->p_list, p_list->len, name, &start, &end );
  //not in list
  if( start != end )
    return -ENOENT;

  //free data
  if( p_list->p_list[end]->p_data != NULL && data_free_func != NULL )
    data_free_func( p_list->p_list[end]->p_data );
  //free list entry
  free( p_list->p_list[end] );

  //decrease length of list
  p_list->len--;
  
  //remove item from list
  for( i = end; i < (int)p_list->len; i++ )
    p_list->p_list[i] = p_list->p_list[i+1];

  //can size be decreased?
  if( p_list->len <= p_list->size )
  {
    //decrease size
    p_list->size -= p_list->size_step;
    if( p_list->size < 16 ) //keep a minimum size of 16 entries
      p_list->size = 16;
    //reallocate list
    p_new_list = (st_list_item * *)realloc( p_list->p_list, p_list->size * sizeof( st_list_item * ) );
    if( p_new_list != NULL ) //if realloc failed, we keep the bigger buffer
      p_list->p_list = p_new_list;
    //must size step be decreased?
    if( p_list->size <= p_list->size_step << 1 )
    {
      p_list->size_step >>= 1;
      if( p_list->size_step < 4 ) //keep a minimum size step of 4 entries
        p_list->size_step = 4;
    }
  }

  return 0;
}

//get the data of an item in a list
//returns the data pointer of NULL on error
void * list_get( st_list * p_list, char * name )
{
  int start, end;

  //truncate name
  if( strlen( name ) >= list_item_name_size )
    name[list_item_name_size-1] = 0;

  //get position
  list_get_pos( p_list->p_list, p_list->len, name, &start, &end );
  //not in list
  if( start != end )
    return NULL;

  //return data pointer
  return p_list->p_list[end]->p_data;
}

//enumerate all items in a list
//the enumerations stops when enum_func retuns a value != 0
void list_enum( st_list * p_list, int (*enum_func)( char *, void *, void * ), void * p_enum_context )
{
  int i;

  //call enumeration function for every item in the list
  for( i = 0; i < (int)p_list->len; i++ )
    if( enum_func( p_list->p_list[i]->name, p_list->p_list[i]->p_data, p_enum_context ) != 0 )
      break;
}
