/* BLINKENLIGHTS PROXY
 * version 0.92 date 2004-04-07
 * Copyright (C) 2003 1stein <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 */

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <malloc.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/ioctl.h>

#include "addr.h"
#include "dynsrc.h"
#include "infilter.h"
#include "list.h"
#include "output.h"
#include "proto.h"
#include "tools.h"
#include "inport.h"

//type for data of input port list items (see header file)
struct s_inport_data
{
  st_addr addr; //the address the input port occupies
  int sock; //the bound socket of the input port
  st_dynsrc_list * p_dynsrc_list; //pointer to the dynamic sources list
  st_infilter_list * p_infilter_list; //pointer to the input filter list
};

//type for an input port list (see header file)
struct s_inport_list
{
  st_list * p_list; //the input port list
};

//create a new input port list data item
//returns the pointer to the data item or NULL in case of error
static st_inport_data * inport_data_new( )
{
  st_inport_data * p_data;

  //allocate a new input port list data item
  p_data = (st_inport_data *)malloc( sizeof( st_inport_data ) );
  if( p_data == NULL )
    return NULL;

  //initialize data item
  p_data->addr.af = AF_UNSPEC; //no address yet
  p_data->addr.sa_len = 0;
  p_data->sock = -1; //no socket yet
  p_data->p_dynsrc_list = NULL; //no dynamic sources list yet
  p_data->p_infilter_list = NULL; //no input filter list yet

  return p_data;
}

//free input port list data item
static void inport_data_free( void * vp_data )
{
  st_inport_data * p_data = vp_data;

  //close socket
  if( p_data->sock != -1 )
    close( p_data->sock );

  //free dynamic sources list
  if( p_data->p_dynsrc_list != NULL )
    dynsrc_free( p_data->p_dynsrc_list );
  
  //free input filter list
  if( p_data->p_infilter_list != NULL )
    infilter_free( p_data->p_infilter_list );

  //free data item
  free( p_data );
}

//create a new input port list
//returns the pointer to the list or NULL in case of error
st_inport_list * inport_new( st_out_info * p_out_info )
{
  st_inport_list * p_inport_list;

  //create structure
  p_inport_list = (st_inport_list *)malloc( sizeof( st_inport_list ) );
  if( p_inport_list == NULL )
  {
    out_str( p_out_info, "  error: could not create new input port list (out of memory)\n" );
    return NULL;
  }

  //create list
  p_inport_list->p_list = list_new( );
  if( p_inport_list->p_list == NULL )
  {
    free( p_inport_list );
    out_str( p_out_info, "  error: could not create new input port list\n" );
    return NULL;
  }

  return p_inport_list;
}

//free an input port list
void inport_free( st_inport_list * p_inport_list )
{
  //free list
  list_free( p_inport_list->p_list, inport_data_free );

  //free structure
  free( p_inport_list );
}

//add an input port to an input port list
//returns 0 on success, a negative value in case of error
int inport_add( st_inport_list * p_inport_list, char * p_name, char * p_addr, st_out_info * p_out_info )
{
  st_inport_data * p_data;
  int ret_val;

  //allocate data item
  p_data = inport_data_new( );
  if( p_data == NULL )
  {
    out_str( p_out_info, "  error: could allocate data item (out of memory)\n" );
    return -ENOMEM;
  }

  //convert address
  if( addr_str2addr( p_addr, &p_data->addr, p_out_info ) != 0 )
  {
    inport_data_free( p_data );
    out_str( p_out_info, "  error: could not parse address\n" );
    return -EINVAL;
  }

  //create socket
  p_data->sock = socket( p_data->addr.af, SOCK_DGRAM, IPPROTO_UDP );
  if( p_data->sock == -1 )
  {
    inport_data_free( p_data );
    out_str( p_out_info, "  error: could not create socket\n" );
    return -EINVAL;
  }
  //bind socket
  if( bind( p_data->sock, &p_data->addr.sa.sa, p_data->addr.sa_len ) == -1 )
  {
    inport_data_free( p_data );
    out_str( p_out_info, "  error: could not bind socket\n" );
    return -EINVAL;
  }

  //create a new dynamic source list
  //(use address family of input port)
  p_data->p_dynsrc_list = dynsrc_new( p_data->addr.af, p_out_info );
  if( p_data->p_dynsrc_list == NULL )
  {
    inport_data_free( p_data );
    out_str( p_out_info, "  error: could not create dynamic sources list\n" );
    return -ENOMEM;
  }

  //create a new input filter list
  //(use address family of input port)
  p_data->p_infilter_list = infilter_new( p_data->addr.af, p_out_info );
  if( p_data->p_infilter_list == NULL )
  {
    inport_data_free( p_data );
    out_str( p_out_info, "  error: could not create input filter list\n" );
    return -ENOMEM;
  }

  //add input port to list
  ret_val = list_add( p_inport_list->p_list, p_name, p_data );
  if( ret_val != 0 )
  {
    inport_data_free( p_data );
    out_str( p_out_info, "  error: could not add input port \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\" to list" );
    switch( ret_val )
    {
      case -EEXIST: out_str( p_out_info, " (already in list)" ); break;
      case -ENOMEM: out_str( p_out_info, " (out of memory)" ); break;
    }
    out_str( p_out_info, "\n" );
    return ret_val;
  }

  return 0;
}

//delete an input port from an input port list
//returns 0 on success, a negative value in case of error
int inport_del( st_inport_list * p_inport_list, char * p_name, st_out_info * p_out_info )
{
  int ret_val;
  
  //delete input port from list
  ret_val = list_del( p_inport_list->p_list, p_name, inport_data_free );
  if( ret_val != 0 )
  {
    out_str( p_out_info, "  error: could not delete input port \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\" from list" );
    switch( ret_val )
    {
      case -ENOENT: out_str( p_out_info, " (not in list)" ); break;
    }
    out_str( p_out_info, "\n" );
    return ret_val;
  }

  return 0;
}

//list input ports - enumeration function
static int inport_list_enum( char * p_name, void * vp_data, void * vp_out_info )
{
  st_inport_data * p_data = vp_data;
  st_out_info * p_out_info = vp_out_info;

  //output name
  out_str( p_out_info, "  " );
  out_str( p_out_info, p_name );
  out_str( p_out_info, "\n" );

  //output address
  out_str( p_out_info, "    address: " );
  addr_addr_out( &p_data->addr, p_out_info );
  out_str( p_out_info, "\n" );

  //continue with enumeration
  return 0;
}

//list input ports in an input port list
void inport_list( st_inport_list * p_inport_list, st_out_info * p_out_info )
{
  //enumerate list items
  list_enum( p_inport_list->p_list, inport_list_enum, (void *)p_out_info );
}

//get input port from list
//returns a pointer to data item or NULL in case of error
st_inport_data * inport_get( st_inport_list * p_inport_list, char * p_name, st_out_info * p_out_info )
{
  st_inport_data * p_data;
  
  //get input port list item
  p_data = list_get( p_inport_list->p_list, p_name );
  if( p_data == NULL )
  {
    out_str( p_out_info, "  error: could not find input port \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\"\n" );
    return NULL;
  }

  //return the pointer to the input port data item
  return p_data;
}

//get dynamic source list of input port
//returns a pointer to the dynamic source list
st_dynsrc_list * inport_dynsrc_list( st_inport_data * p_inport_data )
{
  return p_inport_data->p_dynsrc_list;
}

//get filter list of input port
//returns a pointer to the input filter list
st_infilter_list * inport_infilter_list( st_inport_data * p_inport_data )
{
  return p_inport_data->p_infilter_list;
}

//get socket of input port
//returns the socket fd
int inport_sock( st_inport_data * p_inport_data )
{
  return p_inport_data->sock;
}

//do periodic tasks for input ports - enumeration function
static int inport_tick_enum( char * p_name, void * vp_data, void * vp_unused )
{
  st_inport_data * p_data = vp_data;

  //do periodic tasks for dynamic sources
  dynsrc_tick( p_data->p_dynsrc_list, p_data->sock );

  //continue with enumeration
  return 0;

  //keep compiler happy
  p_name = NULL;
  vp_unused = NULL;
}

//do periodic tasks for input ports
void inport_tick( st_inport_list * p_inport_list )
{
  //enumerate list items
  list_enum( p_inport_list->p_list, inport_tick_enum, NULL );
}

//put sockets in input port list info fd sets - enumeration function
static int inport_fd_sets_enum( char * p_name, void * vp_data, void * vp_fd_sets )
{
  st_inport_data * p_data = vp_data;
  st_select_fd_sets * p_fd_sets = vp_fd_sets;

  //put socket into read fd_set
  if( p_data->sock + 1 > p_fd_sets->n )
    p_fd_sets->n = p_data->sock + 1;
  FD_SET( p_data->sock, &p_fd_sets->read_fd_set );

  //keep compiler happy
  p_name = NULL;

  //continue with enumeration
  return 0;
}

//put sockets in input port list info fd sets
void inport_fd_sets( st_inport_list * p_inport_list, st_select_fd_sets * p_fd_sets )
{
  //enumerate list items
  list_enum( p_inport_list->p_list, inport_fd_sets_enum, (void *)p_fd_sets );
}

//process sockets in input port list that are in fd sets - enumeration function
static int inport_process_enum( char * p_name, void * vp_data, void * vp_fd_sets )
{
  st_inport_data * p_data = vp_data;
  st_select_fd_sets * p_fd_sets = vp_fd_sets;
  int len, i;
  char * p_msg;
  st_addr src_addr;
  st_frame * p_frame;

  //received something on socket
  if( FD_ISSET( p_data->sock, &p_fd_sets->read_fd_set ) )
  {
    //get length of received message
    len = 0;
    if( ioctl( p_data->sock, FIONREAD, &len ) >= 0 )
    {
      if( len > 0 )
      {
        //allocate buffer for message
        p_msg = (char *)malloc( len );
        if( p_msg != NULL )
        {
          //get received message
          src_addr.sa_len = sizeof( src_addr.sa );
          i = recvfrom( p_data->sock, p_msg, len, 0, &src_addr.sa.sa, &src_addr.sa_len );
          if( i != len ) //we've polled the length before, so the same length has to be returned here
            free( p_msg );
          else
          {
            //set up addr structure for source address
            if( addr_sa2addr( &src_addr ) != 0 )
              free( p_msg );
            else
            {
              //parse message and generate a frame from it (with initial usecount of 1)
              p_frame = frame_parse( &src_addr, &p_data->addr, p_msg, len );
              if( p_frame == NULL )
                free( p_msg );
              else
              {
                //pass frame along input filter list
                infilter_process( p_data->p_infilter_list, p_frame );
                //decrement use count for this frame
                frame_dec_use( p_frame );
              }
            }
          }
        }
      }
    }    
  } //if( FD_ISSET( ...

  //continue with enumeration
  return 0;

  //keep compiler happy
  p_name = NULL;
}

//process sockets in input port list that are in fd sets
void inport_process( st_inport_list * p_inport_list, st_select_fd_sets * p_fd_sets )
{
  //enumerate list items
  list_enum( p_inport_list->p_list, inport_process_enum, (void *)p_fd_sets );
}
