/* BLINKENLIGHTS PROXY
 * version 0.100 date 2008-03-21
 * Copyright (C) 2003-2006 Stefan Schuermans <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>

#include "addr.h"
#include "list.h"
#include "output.h"
#include "proto.h"
#include "stream.h"
#include "statdest.h"

//type for data of static destination list items (see header file)
struct s_statdest_data
{
  st_addr addr; //the address to sent the requests to
  et_proto proto; //protocol to use
  st_format format; //format of pictures
};

//type for a static destination list (see header file)
struct s_statdest_list
{
  st_list * p_list; //the static destination list
  int af; //the address family of this list (obtained from owning port)
  st_outport_data * p_outport_data; //pointer to owning outport
};

//create a new static destination list data item
//returns the pointer to the data item or NULL in case of error
static st_statdest_data * statdest_data_new( )
{
  st_statdest_data * p_data;

  //allocate a new static destination list data item
  p_data = (st_statdest_data *)malloc( sizeof( st_statdest_data ) );
  if( p_data == NULL )
    return NULL;

  //initialize data item
  p_data->addr.af = AF_UNSPEC; //no address yet
  p_data->addr.sa_len = 0;
  p_data->proto = proto_none; //no protocol yet
  p_data->format = format_none; //no format yet

  return p_data;
}

//free static destination list data item
static void statdest_data_free( void * vp_data )
{
  st_statdest_data * p_data = vp_data;

  //free data item
  free( p_data );
}

//create a new static destination list
//<af> specifies the address family of this static destination list (obtained from owning port)
//returns the pointer to the list or NULL in case of error
st_statdest_list * statdest_new( int af, st_outport_data * p_outport_data, st_out_info * p_out_info )
{
  st_statdest_list * p_statdest_list;

  //create structure
  p_statdest_list = (st_statdest_list *)malloc( sizeof( st_statdest_list ) );
  if( p_statdest_list == NULL )
  {
    out_str( p_out_info, "  error: could not create new static destination list (out of memory)\n" );
    return NULL;
  }

  //create list
  p_statdest_list->p_list = list_new( );
  if( p_statdest_list->p_list == NULL )
  {
    free( p_statdest_list );
    out_str( p_out_info, "  error: could not create new static destination list\n" );
    return NULL;
  }

  //initialize rest of structure
  p_statdest_list->af = af; //address family of this list must match the address family of the owning port
  p_statdest_list->p_outport_data = p_outport_data; //save pointer to owning outport

  return p_statdest_list;
}

//free a static destination list
void statdest_free( st_statdest_list * p_statdest_list )
{
  //free list
  list_free( p_statdest_list->p_list, statdest_data_free );

  //free structure
  free( p_statdest_list );
}

//add a static destination to a static destination list
//returns 0 on success, a negative value in case of error
int statdest_add( st_statdest_list * p_statdest_list, char * p_name, char * p_addr, char * p_proto, char * p_format, st_out_info * p_out_info )
{
  st_statdest_data * p_data;
  st_stream_list * p_stream_list;
  int ret_val;

  //allocate data item
  p_data = statdest_data_new( );
  if( p_data == NULL )
  {
    out_str( p_out_info, "  error: could allocate data item (out of memory)\n" );
    return -ENOMEM;
  }

  //convert address
  if( addr_str2addr( p_addr, &p_data->addr, p_out_info ) != 0 )
  {
    statdest_data_free( p_data );
    out_str( p_out_info, "  error: could not parse address\n" );
    return -EINVAL;
  }
  //check that address family matches
  if( p_data->addr.af != p_statdest_list->af )
  {
    statdest_data_free( p_data );
    out_str( p_out_info, "  error: address family does not match\n" );
    return -EINVAL;
  }

  //convert protcol
  p_data->proto = proto_str2proto( p_proto, 0 /* proto_any is not allowed */, p_out_info );
  if( p_data->proto == proto_none )
  {
    statdest_data_free( p_data );
    out_str( p_out_info, "  error: could not parse protocol\n" );
    return -EINVAL;
  }

  //convert format
  if( format_str2format( p_format, &p_data->format, p_out_info ) != 0 )
  {
    statdest_data_free( p_data );
    out_str( p_out_info, "  error: could not parse format\n" );
    return -EINVAL;
  }

  //add static destination to list
  ret_val = list_add( p_statdest_list->p_list, p_name, p_data );
  if( ret_val != 0 )
  {
    statdest_data_free( p_data );
    out_str( p_out_info, "  error: could not add static destination \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\" to list" );
    switch( ret_val )
    {
      case -EEXIST: out_str( p_out_info, " (already in list)" ); break;
      case -ENOMEM: out_str( p_out_info, " (out of memory)" ); break;
    }
    out_str( p_out_info, "\n" );
    return ret_val;
  }

  //send current frame to new destination
  p_stream_list = outport_stream_list( p_statdest_list->p_outport_data );
  stream_send_current_frame( p_stream_list, p_data->proto, &p_data->addr, &p_data->format );

  return 0;
}

//delete a static destination from a static destination list
//returns 0 on success, a negative value in case of error
int statdest_del( st_statdest_list * p_statdest_list, char * p_name, st_out_info * p_out_info )
{
  int ret_val;
  
  //delete static destination from list
  ret_val = list_del( p_statdest_list->p_list, p_name, statdest_data_free );
  if( ret_val != 0 )
  {
    out_str( p_out_info, "  error: could not delete static destination \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\" from list" );
    switch( ret_val )
    {
      case -ENOENT: out_str( p_out_info, " (not in list)" ); break;
    }
    out_str( p_out_info, "\n" );
    return ret_val;
  }

  return 0;
}

//list static destinations - enumeration function
static int statdest_list_enum( char * p_name, void * vp_data, void * vp_out_info )
{
  st_statdest_data * p_data = vp_data;
  st_out_info * p_out_info = vp_out_info;

  //output name
  out_str( p_out_info, "  " );
  out_str( p_out_info, p_name );
  out_str( p_out_info, "\n" );

  //output address and protocol
  out_str( p_out_info, "    address: " );
  addr_addr_out( &p_data->addr, p_out_info );
  out_str( p_out_info, "\n    protocol: " );
  proto_proto_out( p_data->proto, p_out_info );
  out_str( p_out_info, "\n    format: " );
  format_format_out( &p_data->format, p_out_info );
  out_str( p_out_info, "\n" );

  //continue with enumeration
  return 0;
}

//list static destinations in a static destination list
void statdest_list( st_statdest_list * p_statdest_list, st_out_info * p_out_info )
{
  //enumerate list items
  list_enum( p_statdest_list->p_list, statdest_list_enum, (void *)p_out_info );
}

//set format of a static destination in a static destination list
//returns 0 on success, a negative value in case of error
int statdest_set_format( st_statdest_list * p_statdest_list, char * p_name, char * p_format, st_out_info * p_out_info )
{
  st_statdest_data * p_data;
  st_format format;
  
  //get static destinatin list item
  p_data = list_get( p_statdest_list->p_list, p_name );
  if( p_data == NULL )
  {
    out_str( p_out_info, "  error: could not find static destination \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\"\n" );
    return -ENOENT;
  }

  //convert format
  if( format_str2format( p_format, &format, p_out_info ) != 0 )
  {
    out_str( p_out_info, "  error: could not parse format\n" );
    return -EINVAL;
  }

  //set new format
  p_data->format = format;

  return 0;
}

//send a frame to static destinations - type for enumeration context
typedef struct s_statdest_send_frame_context
{
  st_frame * p_frame;
  int sock;
} st_statdest_send_frame_context;

//send a frame to static destinations - enumeration function
static int statdest_send_frame_enum( char * p_name, void * vp_data, void * vp_context )
{
  st_statdest_data * p_data = vp_data;
  st_statdest_send_frame_context * p_context = vp_context;

  //send frame
  proto_send_frame( p_context->sock, p_data->proto, &p_data->addr, p_data->format, p_context->p_frame );

  //continue with enumeration
  return 0;

  //keep compiler happy
  p_name = NULL;
}

//send a frame to static destinations
void statdest_send_frame( st_statdest_list * p_statdest_list, st_frame * p_frame, int sock )
{
  st_statdest_send_frame_context context;

  //enumerate list items
  context.p_frame = p_frame;
  context.sock = sock;
  list_enum( p_statdest_list->p_list, statdest_send_frame_enum, (void *)&context );
}
