/* BLINKENLIGHTS PROXY
 * version 0.100 date 2008-03-21
 * Copyright (C) 2003-2006 Stefan Schuermans <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 */

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>

#include "addr.h"
#include "dyndest.h"
#include "global.h"
#include "list.h"
#include "outfilter.h"
#include "output.h"
#include "statdest.h"
#include "stream.h"
#include "tools.h"
#include "outport.h"

//type for data of output port list items (see header file)
struct s_outport_data
{
  st_addr addr; //the address the output port occupies
  int sock; //the bound socket of the output port
  unsigned short timeout_s; //timeout in seconds (0 for none)
  unsigned long last_activity_ms; //time of last activity (global millisecond counter)
  st_stream_list * p_stream_list; //pointer to stream list
  st_statdest_list * p_statdest_list; //pointer to the static destination list
  st_outfilter_list * p_outfilter_list; //pointer to the ouput filter list
  st_dyndest_list * p_dyndest_list; //pointer to dynamic destination list
};

//type for an output port list (see header file)
struct s_outport_list
{
  st_list * p_list; //the output port list
};

//create a new output port list data item
//returns the pointer to the data item or NULL in case of error
static st_outport_data * outport_data_new( )
{
  st_outport_data * p_data;

  //allocate a new output port list data item
  p_data = (st_outport_data *)malloc( sizeof( st_outport_data ) );
  if( p_data == NULL )
    return NULL;

  //initialize data item
  p_data->addr.af = AF_UNSPEC; //no address yet
  p_data->addr.sa_len = 0;
  p_data->sock = -1; //no socket yet
  p_data->timeout_s = 0; //no timeout
  p_data->last_activity_ms = global_millisecs; //not idle
  p_data->p_stream_list = NULL; //no stream list yet
  p_data->p_statdest_list = NULL; //no static destination list yet
  p_data->p_outfilter_list = NULL; //no output filter list yet
  p_data->p_dyndest_list = NULL; //no dynamic destination list yet

  return p_data;
}

//free output port list data item
static void outport_data_free( void * vp_data )
{
  st_outport_data * p_data = vp_data;

  //close socket
  if( p_data->sock != -1 )
    close( p_data->sock );

  //free stream list
  if( p_data->p_stream_list != NULL )
    stream_free( p_data->p_stream_list );

  //free static destination list
  if( p_data->p_statdest_list != NULL )
    statdest_free( p_data->p_statdest_list );

  //free output filter list
  if( p_data->p_outfilter_list != NULL )
    outfilter_free( p_data->p_outfilter_list );

  //free dynamic destination list
  if( p_data->p_dyndest_list != NULL )
    dyndest_free( p_data->p_dyndest_list );

  //free data item
  free( p_data );
}

//create a new output port list
//returns the pointer to the list or NULL in case of error
st_outport_list * outport_new( st_out_info * p_out_info )
{
  st_outport_list * p_outport_list;

  //create structure
  p_outport_list = (st_outport_list *)malloc( sizeof( st_outport_list ) );
  if( p_outport_list == NULL )
  {
    out_str( p_out_info, "  error: could not create new output port list (out of memory)\n" );
    return NULL;
  }

  //create list
  p_outport_list->p_list = list_new( );
  if( p_outport_list->p_list == NULL )
  {
    free( p_outport_list );
    out_str( p_out_info, "  error: could not create new output port list\n" );
    return NULL;
  }

  return p_outport_list;
}

//free an output port list
void outport_free( st_outport_list * p_outport_list )
{
  //free list
  list_free( p_outport_list->p_list, outport_data_free );

  //free structure
  free( p_outport_list );
}

//add an output port to an output port list
//returns 0 on success, a negative value in case of error
int outport_add( st_outport_list * p_outport_list, char * p_name, char * p_addr, char * p_timeout, st_out_info * p_out_info )
{
  st_outport_data * p_data;
  int ret_val;

  //allocate data item
  p_data = outport_data_new( );
  if( p_data == NULL )
  {
    out_str( p_out_info, "  error: could allocate data item (out of memory)\n" );
    return -ENOMEM;
  }

  //convert address
  if( addr_str2addr( p_addr, &p_data->addr, p_out_info ) != 0 )
  {
    outport_data_free( p_data );
    out_str( p_out_info, "  error: could not parse address\n" );
    return -EINVAL;
  }

  //create socket
  p_data->sock = socket( p_data->addr.af, SOCK_DGRAM, IPPROTO_UDP );
  if( p_data->sock == -1 )
  {
    outport_data_free( p_data );
    out_str( p_out_info, "  error: could not create socket\n" );
    return -EINVAL;
  }
  //bind socket
  if( bind( p_data->sock, &p_data->addr.sa.sa, p_data->addr.sa_len ) == -1 )
  {
    outport_data_free( p_data );
    out_str( p_out_info, "  error: could not bind socket\n" );
    return -EINVAL;
  }

  //initialize timeout parameters
  if( sscanf( p_timeout, "%hu", &p_data->timeout_s ) != 1 )
    p_data->timeout_s = 0;
  p_data->last_activity_ms = global_millisecs; //not idle

  //create a new stream list
  p_data->p_stream_list = stream_new( p_data, p_out_info );
  if( p_data->p_stream_list == NULL )
  {
    outport_data_free( p_data );
    out_str( p_out_info, "  error: could not create stream list\n" );
    return -ENOMEM;
  }

  //create a new static destination list
  //(use address family of output port)
  p_data->p_statdest_list = statdest_new( p_data->addr.af, p_data, p_out_info );
  if( p_data->p_statdest_list == NULL )
  {
    outport_data_free( p_data );
    out_str( p_out_info, "  error: could not create static destination list\n" );
    return -ENOMEM;
  }

  //create a new output filter list
  //(use address family of output port)
  p_data->p_outfilter_list = outfilter_new( p_data->addr.af, p_out_info, p_data );
  if( p_data->p_outfilter_list == NULL )
  {
    outport_data_free( p_data );
    out_str( p_out_info, "  error: could not create output filter list\n" );
    return -ENOMEM;
  }

  //create a new dynamic destination list
  p_data->p_dyndest_list = dyndest_new( p_data, p_out_info );
  if( p_data->p_dyndest_list == NULL )
  {
    outport_data_free( p_data );
    out_str( p_out_info, "  error: could not create dynamic destination list\n" );
    return -ENOMEM;
  }

  //add output port to list
  ret_val = list_add( p_outport_list->p_list, p_name, p_data );
  if( ret_val != 0 )
  {
    outport_data_free( p_data );
    out_str( p_out_info, "  error: could not add output port \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\" to list" );
    switch( ret_val )
    {
      case -EEXIST: out_str( p_out_info, " (already in list)" ); break;
      case -ENOMEM: out_str( p_out_info, " (out of memory)" ); break;
    }
    out_str( p_out_info, "\n" );
    return ret_val;
  }

  return 0;
}

//delete an output port from an output port list
//returns 0 on success, a negative value in case of error
int outport_del( st_outport_list * p_outport_list, char * p_name, st_out_info * p_out_info )
{
  int ret_val;
  
  //delete output port from list
  ret_val = list_del( p_outport_list->p_list, p_name, outport_data_free );
  if( ret_val != 0 )
  {
    out_str( p_out_info, "  error: could not delete output port \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\" from list" );
    switch( ret_val )
    {
      case -ENOENT: out_str( p_out_info, " (not in list)" ); break;
    }
    out_str( p_out_info, "\n" );
    return ret_val;
  }

  return 0;
}

//list output ports - enumeration function
static int outport_list_enum( char * p_name, void * vp_data, void * vp_out_info )
{
  st_outport_data * p_data = vp_data;
  st_out_info * p_out_info = vp_out_info;
  char txt[16];

  //output name
  out_str( p_out_info, "  " );
  out_str( p_out_info, p_name );
  out_str( p_out_info, "\n" );

  //output address
  out_str( p_out_info, "    address: " );
  addr_addr_out( &p_data->addr, p_out_info );
  out_str( p_out_info, "\n" );

  //output timeout
  out_str( p_out_info, "    timeout: " );
  snprintf( txt, sizeof( txt ), "%hu", p_data->timeout_s );
  if( p_data->timeout_s == 0 )
    out_str( p_out_info, "none (0)" );
  else
    out_str( p_out_info, txt );
  out_str( p_out_info, " s\n" );

  //continue with enumeration
  return 0;
}

//list output ports in an output port list
void outport_list( st_outport_list * p_outport_list, st_out_info * p_out_info )
{
  //enumerate list items
  list_enum( p_outport_list->p_list, outport_list_enum, (void *)p_out_info );
}

//get output port from list
//returns a pointer to data item or NULL in case of error
st_outport_data * outport_get( st_outport_list * p_outport_list, char * p_name, st_out_info * p_out_info )
{
  st_outport_data * p_data;
  
  //get output port list item
  p_data = list_get( p_outport_list->p_list, p_name );
  if( p_data == NULL )
  {
    out_str( p_out_info, "  error: could not find output port \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\"\n" );
    return NULL;
  }

  //return the pointer to the data item
  return p_data;
}

//get stream list of output port
//returns a pointer to the stream list
st_stream_list * outport_stream_list( st_outport_data * p_outport_data )
{
  return p_outport_data->p_stream_list;
}

//get static destination list of output port
//returns a pointer to the static destination list
st_statdest_list * outport_statdest_list( st_outport_data * p_outport_data )
{
  return p_outport_data->p_statdest_list;
}

//get filter list of output port
//returns a pointer to the ouput filter list
st_outfilter_list * outport_outfilter_list( st_outport_data * p_outport_data )
{
  return p_outport_data->p_outfilter_list;
}

//get socket of output port
//returns the socket fd
int outport_sock( st_outport_data * p_outport_data )
{
  return p_outport_data->sock;
}

//get dynamic destination list of output port
//returns a pointer to the dynamic destination list
st_dyndest_list * outport_dyndest_list( st_outport_data * p_outport_data )
{
  return p_outport_data->p_dyndest_list;
}

//get address of output port
//returns a pointer to the address
st_addr * outport_addr( st_outport_data * p_outport_data )
{
  return &p_outport_data->addr;
}

//do periodic tasks for input ports - enumeration function
static int outport_tick_enum( char * p_name, void * vp_data, void * vp_unused )
{
  st_outport_data * p_data = vp_data;

  //do periodic tasks for streams, e.g. process timeouts
  stream_tick( p_data->p_stream_list );

  //do periodic tasks for dynamic destinations, e.g. process timeouts
  dyndest_tick( p_data->p_dyndest_list );

  //check if idle (and timeout is set)
  unsigned long idle_ms = global_millisecs - p_data->last_activity_ms;
  if( p_data->timeout_s > 0 && idle_ms > (unsigned long)p_data->timeout_s * 1000 )
  {
    //send last frame again
    outport_send_last_frame( p_data );
    //not idle
    p_data->last_activity_ms = global_millisecs;
  }

  //continue with enumeration
  return 0;

  //keep compiler happy
  p_name = NULL;
  vp_unused = NULL;
}

//do periodic tasks for output ports
void outport_tick( st_outport_list * p_outport_list )
{
  //enumerate list items
  list_enum( p_outport_list->p_list, outport_tick_enum, NULL );
}

//put sockets in output port list info fd sets - enumeration function
static int outport_fd_sets_enum( char * p_name, void * vp_data, void * vp_fd_sets )
{
  st_outport_data * p_data = vp_data;
  st_select_fd_sets * p_fd_sets = vp_fd_sets;

  //put socket into read fd_set
  if( p_data->sock + 1 > p_fd_sets->n )
    p_fd_sets->n = p_data->sock + 1;
  FD_SET( p_data->sock, &p_fd_sets->read_fd_set );

  //continue with enumeration
  return 0;

  //keep compiler happy
  p_name = NULL;
}

//put sockets in output port list info fd sets
void outport_fd_sets( st_outport_list * p_outport_list, st_select_fd_sets * p_fd_sets )
{
  //enumerate list items
  list_enum( p_outport_list->p_list, outport_fd_sets_enum, (void *)p_fd_sets );
}

//process sockets in output port list that are in fd sets - enumeration function
static int outport_process_enum( char * p_name, void * vp_data, void * vp_fd_sets )
{
  st_outport_data * p_data = vp_data;
  st_select_fd_sets * p_fd_sets = vp_fd_sets;
  int i, enable;
  char buffer[256];
  st_addr src_addr;
  et_proto proto;

  //received something on socket
  if( FD_ISSET( p_data->sock, &p_fd_sets->read_fd_set ) )
  {
    //get received message
    src_addr.sa_len = sizeof( src_addr.sa );
    i = recvfrom( p_data->sock, buffer, sizeof( buffer ), 0, &src_addr.sa.sa, &src_addr.sa_len );
    if( i > 0 )
    {
      //set up addr structure for source address
      if( addr_sa2addr( &src_addr ) == 0 )
      {
        //parse request
        proto = proto_parse_request( buffer, i, &enable );
        if( proto != proto_none )
          //pass request along output filter list
          outfilter_process( p_data->p_outfilter_list, &src_addr, proto, enable );
      }
    }
  } //if( FD_ISSET( ...

  //continue with enumeration
  return 0;

  //keep compiler happy
  p_name = NULL;
}

//process sockets in output port list that are in fd sets
void outport_process( st_outport_list * p_outport_list, st_select_fd_sets * p_fd_sets )
{
  //enumerate list items
  list_enum( p_outport_list->p_list, outport_process_enum, (void *)p_fd_sets );
}

//send a frame to all destinations of output port
void outport_send_frame( st_outport_data * p_outport_data, st_frame * p_frame )
{
  //send frame to all static destinations
  statdest_send_frame( p_outport_data->p_statdest_list, p_frame, p_outport_data->sock );
  //send frame to all dynamic destinations
  dyndest_send_frame( p_outport_data->p_dyndest_list, p_frame, p_outport_data->sock );
  //not idle
  p_outport_data->last_activity_ms = global_millisecs;
}

//send the last frame to all destinations of output port
void outport_send_last_frame( st_outport_data * p_outport_data )
{
  //get current stream
  st_stream_list * p_stream = p_outport_data->p_stream_list;
  if( p_stream == NULL )
    return;
  //get current frame
  st_frame * p_cur_frame = stream_get_cur_frame( p_stream );
  if( p_cur_frame == NULL )
    return;
  //send current frame to all destinations
  outport_send_frame( p_outport_data, p_cur_frame );
}

