/* BLINKENLIGHTS PROXY
 * version 0.100 date 2008-03-21
 * Copyright (C) 2003-2006 Stefan Schuermans <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <memory.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/time.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>

#include "commands.h"
#include "global.h"
#include "inport.h"
#include "outport.h"
#include "output.h"
#include "tools.h"

#define BL_PROXY_VERSION_STR "version 0.100 date 2008-03-21"

//default listen address for control connections
#define listen_af "in"
#define listen_addr "127.0.0.1"
#define listen_port "4200"

//maximum number of control connections
#define ctrl_conn_cnt_max 5

//global variable to signal end of program
int end = 0;

//signal handler to ignore a signel
void ignore_signal( int sig_no )
{
  sig_no = 0; //keep compiler-happy
}

//signal handler to end the program
void end_signal( int sig_no )
{
  sig_no = 0; //keep compiler-happy
  end = 1;
}

//process input
//returns 1 if the connction should end, 0 otherwise
int process_input( int fd, char * p_buffer, unsigned int buffer_len, unsigned int * p_pos, st_out_info * p_out_info, int out_prompt )
{
  int len, end;
  char dummy[1024];
  char * p_nl;

  //buffer is full
  if( *p_pos >= buffer_len )
  {
    //read from input into dummy buffer
    len = read( fd, dummy, sizeof( dummy ) );
    if( len <= 0 ) //EOF or error
      return 1; //signal end of connection
    //search newline
    p_nl = memchr( dummy, '\n', len );
    if( p_nl == NULL )
      return 0;
    //output error message
    out_str( p_out_info, "  error: line too long\n" );
    //next line
    *p_pos = len - (p_nl + 1 - dummy); //number of characters after newline
    memcpy( p_buffer, p_nl + 1, *p_pos ); //copy them to buffer
    //done
    return 0;
  }

  //read from input
  len = read( fd, p_buffer + *p_pos, buffer_len - *p_pos );
  if( len <= 0 ) //EOF or error
    return 1; //signal end of connection
  *p_pos += len;

  //process all lines that are available now
  for( ; ; )
  {
    //search newline
    p_nl = memchr( p_buffer, '\n', *p_pos );
    if( p_nl == NULL )
      return 0;

    //truncate line at newline
    *p_nl = 0;
  
    //process line
    end = cmd_exec( p_buffer, p_out_info ); //returns != 0 if connection should end

    //next line
    *p_pos = *p_pos - (p_nl + 1 - p_buffer); //number of characters after newline
    memmove( p_buffer, p_nl + 1, *p_pos ); //move them to start of buffer

    //signal connection to end
    if( end )
      return 1;

    //output new prompt
    if( out_prompt )
    {
      out_str( p_out_info, "bl_proxy> " ); //prompt
      out_flush( p_out_info );
    }
  }

  return 0;
}

//main program
int main( int arg_cnt, char * * args )
{
  int local_input, quiet, param_err, i, len, events, conn_end;
  char input_filename[1024], output_filename[1024], * p_input_filename, * p_output_filename;
  char * p_af, * p_addr, * p_port;
  int input_file, output_file;
  int ctrl_listen_sock, ctrl_conn_socks[ctrl_conn_cnt_max];
  char stdin_buffer[1024], input_file_buffer[1024], ctrl_conn_buffer[ctrl_conn_cnt_max][1024];
  unsigned int stdin_pos, input_file_pos, ctrl_conn_pos[ctrl_conn_cnt_max];
  struct sockaddr_in ctrl_sock_addr_in, ctrl_conn_sock_addr_in;
  struct sockaddr_in6 ctrl_sock_addr_in6, ctrl_conn_sock_addr_in6;
  struct sockaddr_un ctrl_sock_addr_un, ctrl_conn_sock_addr_un;
  struct sockaddr * p_ctrl_sock_addr, * p_ctrl_conn_sock_addr;
  int ctrl_sock_addr_len, ctrl_conn_sock_addr_len;
  char * unlink_ctrl_listen_sock;
  st_select_fd_sets fd_sets;
  struct timeval timeout;
  st_out_info output_file_out_info, out_info;

  //generate default filenames for input and output file
  input_filename[0] = 0;
  strncat( input_filename, args[0], sizeof( input_filename ) - 4 );
  strcat( input_filename, ".in" );
  p_input_filename = input_filename;
  output_filename[0] = 0;
  strncat( output_filename, args[0], sizeof( output_filename ) - 5 );
  strcat( output_filename, ".out" );
  p_output_filename = output_filename;

  //parse parameters
  p_af = listen_af;
  p_addr = listen_addr;
  p_port = listen_port;
  local_input = 0;
  quiet = 0;
  param_err = 0;
  for( i = 1; i < arg_cnt; i++ )
  {
    //input file
    if( strcasecmp( args[i], "-i" ) == 0 )
      if( i + 1 < arg_cnt )
        p_input_filename = args[++i];
      else
        param_err = 1;
    else if( strcasecmp( args[i], "-o" ) == 0 )
      if( i + 1 < arg_cnt )
        p_output_filename = args[++i];
      else
        param_err = 1;
    else if( strcasecmp( args[i], "-f" ) == 0 )
      if( i + 1 < arg_cnt )
        p_af = args[++i];
      else
        param_err = 1;
    else if( strcasecmp( args[i], "-a" ) == 0 )
      if( i + 1 < arg_cnt )
        p_addr = args[++i];
      else
        param_err = 1;
    else if( strcasecmp( args[i], "-p" ) == 0 )
      if( i + 1 < arg_cnt )
        p_port = args[++i];
      else
        param_err = 1;
    else if( strcasecmp( args[i], "-l" ) == 0 )
      local_input = 1;
    else if( strcasecmp( args[i], "-q" ) == 0 )
      quiet = 1;
    //parameter error
    else
      param_err = 1;
  }
  
  //print message
  if( ! quiet )
    printf( "\n"
            "BLINKENLIGHTS PROXY\n"
            BL_PROXY_VERSION_STR"\n"
            "Copyright (C) 2003-2006 Stefan Schuermans <1stein@schuermans.info>\n"
            "Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html\n"
	    "\n" );

  //parameter error
  if( param_err )
  {
    printf( "syntax: \"%s [options]\"\n"
            "\n"
            "options: \"-i <input-file>\" (defaults to \"%s\")\n"
            "         \"-o <output-file>\" (defaults to \"%s\")\n"
            "         \"-f <listen-address-family>\" (\"none\", \"in\" or \"in6\", defaults to \""listen_af"\")\n"
            "         \"-a <listen-address>\" (defaults to \""listen_addr"\")\n"
            "         \"-p <listen-port>\" (defaults to \""listen_port"\")\n"
            "         \"-l\" enable local command input\n"
            "         \"-q\" quiet mode\n"
            "\n",
	    args[0], input_filename, output_filename );
    return 1;
  }

  //install signal handler to ignore signals
  signal( SIGPIPE, ignore_signal );
  //install signal handler to end program
  signal( SIGHUP, end_signal );
  signal( SIGINT, end_signal );
  signal( SIGTERM, end_signal );

  //open input file
  input_file = open( p_input_filename, O_RDONLY | O_NOCTTY );
  if( input_file != -1 )
  {
    //open output file
    output_file = open( p_output_filename, O_WRONLY | O_CREAT | O_TRUNC | O_NOCTTY, 0644 );
    //output info
    if( output_file != -1 )
      output_file_out_info = out_info_fd( output_file );
    else
      output_file_out_info = *p_out_none;
  }  
  else
  {
    //no output file
    output_file = -1;
    output_file_out_info = *p_out_none;
  }

  //generate address to listen on for control connections
  if( strcasecmp( p_af, "none" ) == 0 )
  {
    p_ctrl_sock_addr = NULL;
	ctrl_sock_addr_len = 0;
    p_ctrl_conn_sock_addr = NULL;
    ctrl_conn_sock_addr_len = 0;
	unlink_ctrl_listen_sock = NULL;
  }
  else if( strcasecmp( p_af, "in" ) == 0 )
  {
    ctrl_sock_addr_in.sin_family = AF_INET;
    ctrl_sock_addr_in.sin_port = htons( (uint16_t)atoi( p_port ) );
    if( ! inet_pton( AF_INET, p_addr, &ctrl_sock_addr_in.sin_addr ) )
    {
      if( input_file != -1 )
        close( input_file );
      if( output_file != -1 )
        close( output_file );
      printf( "invalid IN-address: %s\n\n", p_addr );
      return -1;
    }
    p_ctrl_sock_addr = (struct sockaddr *)&ctrl_sock_addr_in;
    ctrl_sock_addr_len = sizeof( ctrl_sock_addr_in );
    p_ctrl_conn_sock_addr = (struct sockaddr *)&ctrl_conn_sock_addr_in;
    ctrl_conn_sock_addr_len = sizeof( ctrl_conn_sock_addr_in );
	unlink_ctrl_listen_sock = NULL;
  }
  else if( strcasecmp( p_af, "in6" ) == 0 )
  {
    ctrl_sock_addr_in6.sin6_family = AF_INET6;
    ctrl_sock_addr_in6.sin6_port = htons( atoi( p_port ) );
    if( ! inet_pton( AF_INET6, p_addr, &ctrl_sock_addr_in6.sin6_addr ) )
    {
      if( input_file != -1 )
        close( input_file );
      if( output_file != -1 )
        close( output_file );
      printf( "invalid IN6-address: %s\n\n", p_addr );
      return -1;
    }
    p_ctrl_sock_addr = (struct sockaddr *)&ctrl_sock_addr_in6;
    ctrl_sock_addr_len = sizeof( ctrl_sock_addr_in6 );
    p_ctrl_conn_sock_addr = (struct sockaddr *)&ctrl_conn_sock_addr_in6;
    ctrl_conn_sock_addr_len = sizeof( ctrl_conn_sock_addr_in6 );
	unlink_ctrl_listen_sock = NULL;
  }
  else if( strcasecmp( p_af, "unix" ) == 0 )
  {
    ctrl_sock_addr_un.sun_family = AF_UNIX;
	if( strlen( p_addr ) > sizeof( ctrl_sock_addr_un.sun_path ) - 1 )
	{
      if( input_file != -1 )
        close( input_file );
      if( output_file != -1 )
        close( output_file );
      printf( "invalid UNIX-address (too long): %s\n\n", p_addr );
      return -1;
	}
    strcpy( ctrl_sock_addr_un.sun_path, p_addr );
    p_ctrl_sock_addr = (struct sockaddr *)&ctrl_sock_addr_un;
    ctrl_sock_addr_len = sizeof( ctrl_sock_addr_un );
    p_ctrl_conn_sock_addr = (struct sockaddr *)&ctrl_conn_sock_addr_un;
    ctrl_conn_sock_addr_len = sizeof( ctrl_conn_sock_addr_un );
	unlink_ctrl_listen_sock = p_addr;
  }
  else
  {
    if( input_file != -1 )
      close( input_file );
    if( output_file != -1 )
      close( output_file );
    printf( "invalid address family \"%s\" for control connections (try \"none\", \"in\", \"in6\" or \"unix\")\n\n", p_af );
    return -1;
  }

  //create socket for control connections
  if( p_ctrl_sock_addr == NULL || p_ctrl_conn_sock_addr == NULL )
  {
    ctrl_listen_sock = -1; //control connections disabled
  }
  else
  {
    ctrl_listen_sock = socket( p_ctrl_sock_addr->sa_family, SOCK_STREAM, 0 );
    if( ctrl_listen_sock == -1 )
    {
      if( input_file != -1 )
        close( input_file );
      if( output_file != -1 )
        close( output_file );
      printf( "could not create socket for control connections: %s\n\n", strerror( errno ) );
      return -1;
    }
  }

  if( ctrl_listen_sock != -1 )
  {
    //bind socket for control connections
    if( bind( ctrl_listen_sock, p_ctrl_sock_addr, ctrl_sock_addr_len ) == -1 )
    {
      close( ctrl_listen_sock );
      if( input_file != -1 )
        close( input_file );
      if( output_file != -1 )
        close( output_file );
      printf( "could not bind socket to \"%s %s %s\" for control connections: %s\n\n", p_af, p_addr, p_port, strerror( errno ) );
      return -1;
    }

    //listen on socket for control connections
    if( listen( ctrl_listen_sock, ctrl_conn_cnt_max ) == -1 )
    {
      close( ctrl_listen_sock );
      if( unlink_ctrl_listen_sock != NULL )
        unlink( unlink_ctrl_listen_sock );
      if( input_file != -1 )
        close( input_file );
      if( output_file != -1 )
        close( output_file );
      printf( "could not listen on socket for control connections: %s\n\n", strerror( errno ) );
      return -1;
    }
    for( i = 0; i < ctrl_conn_cnt_max; i++ )
      ctrl_conn_socks[i] = -1;
  }

  //initialize global stuff
  if( global_init( p_out_stdout ) != 0 )
  {
    if( ctrl_listen_sock != -1 )
    {
      close( ctrl_listen_sock );
      if( unlink_ctrl_listen_sock != NULL )
        unlink( unlink_ctrl_listen_sock );
    }
    if( input_file != -1 )
      close( input_file );
    if( output_file != -1 )
      close( output_file );
    printf( "  error: could not initialize\n\n" );
    return -1;
  }

  //output first prompt
  if( local_input )
  {
    out_str( p_out_stdout, "\n" );
    out_str( p_out_stdout, "bl_proxy> " ); //prompt
    out_flush( p_out_stdout );
  }

  //main loop
  stdin_pos = 0;
  input_file_pos = 0;
  for( ; ! end; )
  {
    //do periodic global tasks (must be done before all other global tasks)
    global_tick( );
    //do periodic tasks for input and output ports
    inport_tick( p_global_inport_list );
    outport_tick( p_global_outport_list );

    //initialize fd_sets
    fd_sets.n = 0;
    FD_ZERO( &fd_sets.read_fd_set );
    FD_ZERO( &fd_sets.write_fd_set );
    FD_ZERO( &fd_sets.except_fd_set );
    //add sockets of input and output ports to fd_sets
    inport_fd_sets( p_global_inport_list, &fd_sets );
    outport_fd_sets( p_global_outport_list, &fd_sets );
    //add stdin to fd_set for reading
    if( local_input )
    {
      FD_SET( 0, &fd_sets.read_fd_set );
      if( 0 + 1 > fd_sets.n )
        fd_sets.n = 0 + 1;
    }
    //add input file
    if( input_file != -1 )
    {
      FD_SET( input_file, &fd_sets.read_fd_set );
      if( input_file + 1 > fd_sets.n )
        fd_sets.n = input_file + 1;
    }
    //control connections enabled
    if( ctrl_listen_sock != -1 )
    {
      //add control connection sockets
      for( i = 0; i < ctrl_conn_cnt_max; i++ ) //check if place for new connection
        if( ctrl_conn_socks[i] == -1 )
          break;
      if( i < ctrl_conn_cnt_max ) //place for new connection is available
      {
        FD_SET( ctrl_listen_sock, &fd_sets.read_fd_set );
        if( ctrl_listen_sock + 1 > fd_sets.n )
          fd_sets.n = ctrl_listen_sock + 1;
      }
      for( i = 0; i < ctrl_conn_cnt_max; i++ ) //established connections
      {
        if( ctrl_conn_socks[i] != -1 )
        {
          FD_SET( ctrl_conn_socks[i], &fd_sets.read_fd_set );
          if( ctrl_conn_socks[i] + 1 > fd_sets.n )
            fd_sets.n = ctrl_conn_socks[i] + 1;
        }
      }
    } //control connections enabled

    //initialize timeout
    timeout.tv_sec = 0;
    timeout.tv_usec = 50000;

    //wait for events
    events = select( fd_sets.n, &fd_sets.read_fd_set, &fd_sets.write_fd_set, &fd_sets.except_fd_set, &timeout );
    if( events < 0 ) //error
      end = 1;
    if( events > 0 ) //at least one event
    {
      //events on sockets of input and output ports
      inport_process( p_global_inport_list, &fd_sets );
      outport_process( p_global_outport_list, &fd_sets );
      //input on stdin
      if( local_input && FD_ISSET( 0, &fd_sets.read_fd_set ) )
      {
        conn_end = process_input( 0, stdin_buffer, sizeof( stdin_buffer ), &stdin_pos, p_out_stdout, 1 ); //returns !=0 if program should end
        if( conn_end )
          end = 1;
      }
      //input on input file
      if( input_file != -1  && FD_ISSET( input_file, &fd_sets.read_fd_set ) )
      {
        conn_end = process_input( input_file, input_file_buffer, sizeof( input_file_buffer ), &input_file_pos, &output_file_out_info, 0 ); //returns !=0 if file processing should stop
        if( conn_end ) //close input and output file
        {
          close( input_file );
          input_file = -1;
          if( output_file != -1 )
          {
            close( output_file );
            output_file = -1;
          }
          output_file_out_info = *p_out_none;
        }
      }
      //control connections enabled
      if( ctrl_listen_sock != -1 )
      {
        //pending connection on control socket
        if( FD_ISSET( ctrl_listen_sock, &fd_sets.read_fd_set ) )
        {
          //search free place
          for( i = 0; i < ctrl_conn_cnt_max; i++ )
            if( ctrl_conn_socks[i] == -1 )
              break;
          if( i < ctrl_conn_cnt_max )
          {
            //accept connection
            len = ctrl_conn_sock_addr_len;
            ctrl_conn_socks[i] = accept( ctrl_listen_sock, p_ctrl_conn_sock_addr, (unsigned int *)&len );
            if( ctrl_conn_socks[i] != -1 )
            {
              //clear buffer
              ctrl_conn_pos[i] = 0;
              //output message and first prompt
              out_info = out_info_fd( ctrl_conn_socks[i] );
              out_str( &out_info, "BLINKENLIGHTS PROXY\n"
                                  BL_PROXY_VERSION_STR"\n"
                                  "\n"
                                  "bl_proxy> " );
              out_flush( &out_info );
            }
          }
        }
        //input on control connection
        for( i = 0; i < ctrl_conn_cnt_max; i++ )
        {
          if( ctrl_conn_socks[i] != -1 )
          {
            if( FD_ISSET( ctrl_conn_socks[i], &fd_sets.read_fd_set ) )
            {
              //process input
              out_info = out_info_fd( ctrl_conn_socks[i] );
              conn_end = process_input( ctrl_conn_socks[i], ctrl_conn_buffer[i], sizeof( ctrl_conn_buffer[i] ), &ctrl_conn_pos[i], &out_info, 1 ); //returns !=0 if connection should end
              //close connection
              if( conn_end )
              {
                shutdown( ctrl_conn_socks[i], SHUT_RDWR );
                close( ctrl_conn_socks[i] );
                ctrl_conn_socks[i] = -1;
              }
            }
          }
        }
      } //control connections enabled
    } //at least one event
  } //for( end = 0; ! end; )
  if( ! quiet )
    printf( "\n" );

  //cleanup global stuff
  global_exit( );

  //close control conntections
  for( i = 0; i < ctrl_conn_cnt_max; i++ )
  {
    if( ctrl_conn_socks[i] != -1 )
    {
      shutdown( ctrl_conn_socks[i], SHUT_RDWR );
      close( ctrl_conn_socks[i] );
    }
  }
  //close listening socket for control connections
  if( ctrl_listen_sock != -1 )
  {
    close( ctrl_listen_sock );
    if( unlink_ctrl_listen_sock != NULL )
      unlink( unlink_ctrl_listen_sock );
  }
  //close input and output file
  if( input_file != -1 )
    close( input_file );
  if( output_file != -1 )
    close( output_file );

  return 0;
}
