/* BLINKENLIGHTS PROXY
 * version 0.100 date 2008-03-21
 * Copyright (C) 2003-2006 Stefan Schuermans <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 */

#include <string.h>

#include "inport.h"
#include "dynsrc.h"
#include "global.h"
#include "output.h"
#include "stream.h"
#include "tools.h"
#include "commands.h"

//help command protoype
void cmd_help( char * p_params, st_out_info * p_out_info );

//list dynamic destinations of an output port
void cmd_dyndest_list( char * p_params, st_out_info * p_out_info )
{
  char * p_op_name;
  st_outport_data * p_outport_data;
  st_dyndest_list * p_dyndest_list;

  //get output port name
  tools_str_first_words( p_params, 1, &p_op_name, &p_params );

  //get output port
  p_outport_data = outport_get( p_global_outport_list, p_op_name, p_out_info );
  if( p_outport_data == NULL )
  {
    out_str( p_out_info, "  error: could not get output port \"" );
    out_str( p_out_info, p_op_name );
    out_str( p_out_info, "\"\n" );
    return;
  }

  //get dynamic destination list of output port
  p_dyndest_list = outport_dyndest_list( p_outport_data );

  //list dynamic destinations
  dyndest_list( p_dyndest_list, p_out_info );
}

//add a dynamic source to an input port
void cmd_dynsrc_list_add( char * p_params, st_out_info * p_out_info )
{
  char * p_ip_name, * p_name, * p_addr, * p_proto, * p_interval;
  st_inport_data * p_inport_data;
  st_dynsrc_list * p_dynsrc_list;
  int sock;
  
  //get input port name, name, address, protocol, interval
  tools_str_first_words( p_params, 1, &p_ip_name, &p_params );
  tools_str_first_words( p_params, 1, &p_name, &p_params );
  tools_str_first_words( p_params, 3, &p_addr, &p_params );
  tools_str_first_words( p_params, 1, &p_proto, &p_params );
  tools_str_first_words( p_params, 1, &p_interval, &p_params );

  //get input port
  p_inport_data = inport_get( p_global_inport_list, p_ip_name, p_out_info );
  if( p_inport_data == NULL )
  {
    out_str( p_out_info, "  error: could not get input port \"" );
    out_str( p_out_info, p_ip_name );
    out_str( p_out_info, "\"\n" );
    return;
  }

  //get dynamic source list of input port
  p_dynsrc_list = inport_dynsrc_list( p_inport_data );

  //get socket of input port
  sock = inport_sock( p_inport_data );

  //add dynamic source
  if( dynsrc_add( p_dynsrc_list, p_name, p_addr, p_proto, p_interval, sock, p_out_info ) != 0 )
  {
    out_str( p_out_info, "  error: could not add dynamic source \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\" to input port \"" );
    out_str( p_out_info, p_ip_name );
    out_str( p_out_info, "\"\n" );
  }
}

//delete a dynamic source from an input port
void cmd_dynsrc_list_del( char * p_params, st_out_info * p_out_info )
{
  char * p_ip_name, * p_name;
  st_inport_data * p_inport_data;
  st_dynsrc_list * p_dynsrc_list;
  int sock;

  //get input port name and name
  tools_str_first_words( p_params, 1, &p_ip_name, &p_params );
  tools_str_first_words( p_params, 1, &p_name, &p_params );

  //get input port
  p_inport_data = inport_get( p_global_inport_list, p_ip_name, p_out_info );
  if( p_inport_data == NULL )
  {
    out_str( p_out_info, "  error: could not get input port \"" );
    out_str( p_out_info, p_ip_name );
    out_str( p_out_info, "\"\n" );
    return;
  }

  //get dynamic source list of input port
  p_dynsrc_list = inport_dynsrc_list( p_inport_data );

  //get socket of input port
  sock = inport_sock( p_inport_data );

  //delete dynamic source
  if( dynsrc_del( p_dynsrc_list, p_name, sock, p_out_info ) != 0 )
  {
    out_str( p_out_info, "  error: could not delete dynamic source \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\" from input port \"" );
    out_str( p_out_info, p_ip_name );
    out_str( p_out_info, "\"\n" );
  }
}

//list dynamic sources of an input port
void cmd_dynsrc_list( char * p_params, st_out_info * p_out_info )
{
  char * p_ip_name;
  st_inport_data * p_inport_data;
  st_dynsrc_list * p_dynsrc_list;

  //get input port name
  tools_str_first_words( p_params, 1, &p_ip_name, &p_params );

  //get input port
  p_inport_data = inport_get( p_global_inport_list, p_ip_name, p_out_info );
  if( p_inport_data == NULL )
  {
    out_str( p_out_info, "  error: could not get input port \"" );
    out_str( p_out_info, p_ip_name );
    out_str( p_out_info, "\"\n" );
    return;
  }

  //get dynamic source list of input port
  p_dynsrc_list = inport_dynsrc_list( p_inport_data );

  //list dynamic sources
  dynsrc_list( p_dynsrc_list, p_out_info );
}

//add an input filter rule to an input port
void cmd_infilter_list_add( char * p_params, st_out_info * p_out_info )
{
  char * p_ip_name, * p_index, * p_range, * p_proto, * p_format, * p_name, * p_stop, * p_op_name, * p_priority, * p_timeout;
  char stop;
  st_inport_data * p_inport_data;
  st_infilter_list * p_infilter_list;
  
  //get input port name, index, address range, protocol, format, name, stop flag, output port name, priority, timeout
  tools_str_first_words( p_params, 1, &p_ip_name, &p_params );
  tools_str_first_words( p_params, 1, &p_index, &p_params );
  tools_str_first_words( p_params, 3, &p_range, &p_params );
  tools_str_first_words( p_params, 1, &p_proto, &p_params );
  tools_str_first_words( p_params, 1, &p_format, &p_params );
  tools_str_first_words( p_params, 1, &p_name, &p_params );
  tools_str_first_words( p_params, 1, &p_stop, &p_params );
  stop = strcasecmp( p_stop, "stop" ) == 0;
  if( stop ) //if stop flag is set
    tools_str_first_words( p_params, 1, &p_op_name, &p_params ); //get next parameter
  else //if stop flag is not set
    p_op_name = p_stop; //use parameter that was not stop flag
  tools_str_first_words( p_params, 1, &p_priority, &p_params );
  tools_str_first_words( p_params, 1, &p_timeout, &p_params );
  if( stop && *p_op_name == 0 ) //if stop flag is there and no output port name is given
  {
    p_priority = "0"; //set priority to 0 to allow adding input filter rule
    p_timeout = "0"; //set timeout to 0 to allow adding input filter rule
  }

  //get input port
  p_inport_data = inport_get( p_global_inport_list, p_ip_name, p_out_info );
  if( p_inport_data == NULL )
  {
    out_str( p_out_info, "  error: could not get input port \"" );
    out_str( p_out_info, p_ip_name );
    out_str( p_out_info, "\"\n" );
    return;
  }

  //get input filter list list of input port
  p_infilter_list = inport_infilter_list( p_inport_data );

  //add input filter rule
  if( infilter_add( p_infilter_list, p_index, p_range, p_proto, p_format, p_name, stop, p_op_name, p_priority, p_timeout, p_out_info ) != 0 )
  {
    out_str( p_out_info, "  error: could not add input filter rule \"" );
    out_str( p_out_info, p_index );
    out_str( p_out_info, "\" to input port \"" );
    out_str( p_out_info, p_ip_name );
    out_str( p_out_info, "\"\n" );
  }
}

//delete an input filter rule from an input port
void cmd_infilter_list_del( char * p_params, st_out_info * p_out_info )
{
  char * p_ip_name, * p_index;
  st_inport_data * p_inport_data;
  st_infilter_list * p_infilter_list;

  //get input port name and index
  tools_str_first_words( p_params, 1, &p_ip_name, &p_params );
  tools_str_first_words( p_params, 1, &p_index, &p_params );

  //get input port
  p_inport_data = inport_get( p_global_inport_list, p_ip_name, p_out_info );
  if( p_inport_data == NULL )
  {
    out_str( p_out_info, "  error: could not get input port \"" );
    out_str( p_out_info, p_ip_name );
    out_str( p_out_info, "\"\n" );
    return;
  }

  //get input filter list list of input port
  p_infilter_list = inport_infilter_list( p_inport_data );

  //delete input filter rule
  if( infilter_del( p_infilter_list, p_index, p_out_info ) != 0 )
  {
    out_str( p_out_info, "  error: could not delete input filter rule \"" );
    out_str( p_out_info, p_index );
    out_str( p_out_info, "\" from input port \"" );
    out_str( p_out_info, p_ip_name );
    out_str( p_out_info, "\"\n" );
  }
}

//list input filter rules of an input port
void cmd_infilter_list( char * p_params, st_out_info * p_out_info )
{
  char * p_ip_name;
  st_inport_data * p_inport_data;
  st_infilter_list * p_infilter_list;

  //get input port name
  tools_str_first_words( p_params, 1, &p_ip_name, &p_params );

  //get input port
  p_inport_data = inport_get( p_global_inport_list, p_ip_name, p_out_info );
  if( p_inport_data == NULL )
  {
    out_str( p_out_info, "  error: could not get input port \"" );
    out_str( p_out_info, p_ip_name );
    out_str( p_out_info, "\"\n" );
    return;
  }

  //get input filter list list of input port
  p_infilter_list = inport_infilter_list( p_inport_data );

  //list input filter rules
  infilter_list( p_infilter_list, p_out_info );
}

//add an input port
void cmd_inport_list_add( char * p_params, st_out_info * p_out_info )
{
  char * p_name, * p_addr;

  //get name and address
  tools_str_first_words( p_params, 1, &p_name, &p_params );
  tools_str_first_words( p_params, 3, &p_addr, &p_params );

  //add input port
  if( inport_add( p_global_inport_list, p_name, p_addr, p_out_info ) != 0 )
  {
    out_str( p_out_info, "  error: could not add input port \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\"\n" );
  }
}

//delete an input port
void cmd_inport_list_del( char * p_params, st_out_info * p_out_info )
{
  char * p_name;

  //get name
  tools_str_first_words( p_params, 1, &p_name, &p_params );

  //delete input port
  if( inport_del( p_global_inport_list, p_name, p_out_info ) != 0 )
  {
    out_str( p_out_info, "  error: could not delete input port \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\"\n" );
  }
}

//list input ports
void cmd_inport_list( char * p_params, st_out_info * p_out_info )
{
  //list input ports
  inport_list( p_global_inport_list, p_out_info );

  //keep compiler happy
  p_params = NULL;
}

//add an output filter rule to an output port
void cmd_outfilter_list_add( char * p_params, st_out_info * p_out_info )
{
  char * p_op_name, * p_index, * p_range, * p_proto, * p_name, * p_timeout, * p_format;
  char deny;
  st_outport_data * p_outport_data;
  st_outfilter_list * p_outfilter_list;
  
  //get output port name, index, address range, protocol, name, deny flag, timeout, format
  tools_str_first_words( p_params, 1, &p_op_name, &p_params );
  tools_str_first_words( p_params, 1, &p_index, &p_params );
  tools_str_first_words( p_params, 3, &p_range, &p_params );
  tools_str_first_words( p_params, 1, &p_proto, &p_params );
  tools_str_first_words( p_params, 1, &p_name, &p_params );
  tools_str_first_words( p_params, 1, &p_timeout, &p_params );
  deny = strcasecmp( p_timeout, "deny" ) == 0;
  tools_str_first_words( p_params, 1, &p_format, &p_params );

  //get output port
  p_outport_data = outport_get( p_global_outport_list, p_op_name, p_out_info );
  if( p_outport_data == NULL )
  {
    out_str( p_out_info, "  error: could not get output port \"" );
    out_str( p_out_info, p_op_name );
    out_str( p_out_info, "\"\n" );
    return;
  }

  //get output filter list list of output port
  p_outfilter_list = outport_outfilter_list( p_outport_data );

  //add output filter rule
  if( outfilter_add( p_outfilter_list, p_index, p_range, p_proto, p_name, deny, p_timeout, p_format, p_out_info ) != 0 )
  {
    out_str( p_out_info, "  error: could not add output filter rule \"" );
    out_str( p_out_info, p_index );
    out_str( p_out_info, "\" to output port \"" );
    out_str( p_out_info, p_op_name );
    out_str( p_out_info, "\"\n" );
  }
}

//delete an output filter rule from an output port
void cmd_outfilter_list_del( char * p_params, st_out_info * p_out_info )
{
  char * p_op_name, * p_index;
  st_outport_data * p_outport_data;
  st_outfilter_list * p_outfilter_list;

  //get output port name and index
  tools_str_first_words( p_params, 1, &p_op_name, &p_params );
  tools_str_first_words( p_params, 1, &p_index, &p_params );

  //get output port
  p_outport_data = outport_get( p_global_outport_list, p_op_name, p_out_info );
  if( p_outport_data == NULL )
  {
    out_str( p_out_info, "  error: could not get output port \"" );
    out_str( p_out_info, p_op_name );
    out_str( p_out_info, "\"\n" );
    return;
  }

  //get output filter list list of output port
  p_outfilter_list = outport_outfilter_list( p_outport_data );

  //delete output filter rule
  if( outfilter_del( p_outfilter_list, p_index, p_out_info ) != 0 )
  {
    out_str( p_out_info, "  error: could not delete output filter rule \"" );
    out_str( p_out_info, p_index );
    out_str( p_out_info, "\" from output port \"" );
    out_str( p_out_info, p_op_name );
    out_str( p_out_info, "\"\n" );
  }
}

//list output filter rules of an output port
void cmd_outfilter_list( char * p_params, st_out_info * p_out_info )
{
  char * p_op_name;
  st_outport_data * p_outport_data;
  st_outfilter_list * p_outfilter_list;

  //get output port name
  tools_str_first_words( p_params, 1, &p_op_name, &p_params );

  //get output port
  p_outport_data = outport_get( p_global_outport_list, p_op_name, p_out_info );
  if( p_outport_data == NULL )
  {
    out_str( p_out_info, "  error: could not get output port \"" );
    out_str( p_out_info, p_op_name );
    out_str( p_out_info, "\"\n" );
    return;
  }

  //get filter list of output port
  p_outfilter_list = outport_outfilter_list( p_outport_data );

  //list output filter rules
  outfilter_list( p_outfilter_list, p_out_info );
}

//add an output port
void cmd_outport_list_add( char * p_params, st_out_info * p_out_info )
{
  char * p_name, * p_addr, * p_timeout;

  //get name and address
  tools_str_first_words( p_params, 1, &p_name, &p_params );
  tools_str_first_words( p_params, 3, &p_addr, &p_params );
  tools_str_first_words( p_params, 1, &p_timeout, &p_params );

  //add output port
  if( outport_add( p_global_outport_list, p_name, p_addr, p_timeout, p_out_info ) != 0 )
  {
    out_str( p_out_info, "  error: could not add output port \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\"\n" );
  }
}

//delete an output port
void cmd_outport_list_del( char * p_params, st_out_info * p_out_info )
{
  char * p_name;

  //get name
  tools_str_first_words( p_params, 1, &p_name, &p_params );

  //delete output port
  if( outport_del( p_global_outport_list, p_name, p_out_info ) != 0 )
  {
    out_str( p_out_info, "  error: could not delete output port \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\"\n" );
  }
}

//list output ports
void cmd_outport_list( char * p_params, st_out_info * p_out_info )
{
  //list output ports
  outport_list( p_global_outport_list, p_out_info );

  //keep compiler happy
  p_params = NULL;
}

//add a static destination to an output port
void cmd_statdest_list_add( char * p_params, st_out_info * p_out_info )
{
  char * p_op_name, * p_name, * p_addr, * p_proto, * p_format;
  st_outport_data * p_outport_data;
  st_statdest_list * p_statdest_list;
  
  //get output port name, name, address, protocol
  tools_str_first_words( p_params, 1, &p_op_name, &p_params );
  tools_str_first_words( p_params, 1, &p_name, &p_params );
  tools_str_first_words( p_params, 3, &p_addr, &p_params );
  tools_str_first_words( p_params, 1, &p_proto, &p_params );
  tools_str_first_words( p_params, 1, &p_format, &p_params );

  //get output port
  p_outport_data = outport_get( p_global_outport_list, p_op_name, p_out_info );
  if( p_outport_data == NULL )
  {
    out_str( p_out_info, "  error: could not get output port \"" );
    out_str( p_out_info, p_op_name );
    out_str( p_out_info, "\"\n" );
    return;
  }

  //get static destination list of output port
  p_statdest_list = outport_statdest_list( p_outport_data );

  //add static destination
  if( statdest_add( p_statdest_list, p_name, p_addr, p_proto, p_format, p_out_info ) != 0 )
  {
    out_str( p_out_info, "  error: could not add static destination \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\" to output port \"" );
    out_str( p_out_info, p_op_name );
    out_str( p_out_info, "\"\n" );
  }
}

//delete a static destination from an output port
void cmd_statdest_list_del( char * p_params, st_out_info * p_out_info )
{
  char * p_op_name, * p_name;
  st_outport_data * p_outport_data;
  st_statdest_list * p_statdest_list;

  //get output port name and name
  tools_str_first_words( p_params, 1, &p_op_name, &p_params );
  tools_str_first_words( p_params, 1, &p_name, &p_params );

  //get output port
  p_outport_data = outport_get( p_global_outport_list, p_op_name, p_out_info );
  if( p_outport_data == NULL )
  {
    out_str( p_out_info, "  error: could not get output port \"" );
    out_str( p_out_info, p_op_name );
    out_str( p_out_info, "\"\n" );
    return;
  }

  //get static destination list of output port
  p_statdest_list = outport_statdest_list( p_outport_data );

  //delete static destination
  if( statdest_del( p_statdest_list, p_name, p_out_info ) != 0 )
  {
    out_str( p_out_info, "  error: could not delete static destination \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\" from output port \"" );
    out_str( p_out_info, p_op_name );
    out_str( p_out_info, "\"\n" );
  }
}

//set new picture format for a static destination of an output port
void cmd_statdest_list_format( char * p_params, st_out_info * p_out_info )
{
  char * p_op_name, * p_name, * p_format;
  st_outport_data * p_outport_data;
  st_statdest_list * p_statdest_list;

  //get output port name, name and format
  tools_str_first_words( p_params, 1, &p_op_name, &p_params );
  tools_str_first_words( p_params, 1, &p_name, &p_params );
  tools_str_first_words( p_params, 1, &p_format, &p_params );

  //get output port
  p_outport_data = outport_get( p_global_outport_list, p_op_name, p_out_info );
  if( p_outport_data == NULL )
  {
    out_str( p_out_info, "  error: could not get output port \"" );
    out_str( p_out_info, p_op_name );
    out_str( p_out_info, "\"\n" );
    return;
  }

  //get static destination list of output port
  p_statdest_list = outport_statdest_list( p_outport_data );

  //set new format for static destination
  if( statdest_set_format( p_statdest_list, p_name, p_format, p_out_info ) != 0 )
  {
    out_str( p_out_info, "  error: could not set format of static destination \"" );
    out_str( p_out_info, p_name );
    out_str( p_out_info, "\" of output port \"" );
    out_str( p_out_info, p_op_name );
    out_str( p_out_info, "\"\n" );
  }
}

//list static destinations of an output port
void cmd_statdest_list( char * p_params, st_out_info * p_out_info )
{
  char * p_op_name;
  st_outport_data * p_outport_data;
  st_statdest_list * p_statdest_list;

  //get output port name
  tools_str_first_words( p_params, 1, &p_op_name, &p_params );

  //get output port
  p_outport_data = outport_get( p_global_outport_list, p_op_name, p_out_info );
  if( p_outport_data == NULL )
  {
    out_str( p_out_info, "  error: could not get output port \"" );
    out_str( p_out_info, p_op_name );
    out_str( p_out_info, "\"\n" );
    return;
  }

  //get static destination list of output port
  p_statdest_list = outport_statdest_list( p_outport_data );

  //list static destinations
  statdest_list( p_statdest_list, p_out_info );
}

//list streams of an output port
void cmd_stream_list( char * p_params, st_out_info * p_out_info )
{
  char * p_op_name;
  st_outport_data * p_outport_data;
  st_stream_list * p_stream_list;

  //get output port name
  tools_str_first_words( p_params, 1, &p_op_name, &p_params );

  //get output port
  p_outport_data = outport_get( p_global_outport_list, p_op_name, p_out_info );
  if( p_outport_data == NULL )
  {
    out_str( p_out_info, "  error: could not get output port \"" );
    out_str( p_out_info, p_op_name );
    out_str( p_out_info, "\"\n" );
    return;
  }

  //get stream list of output port
  p_stream_list = outport_stream_list( p_outport_data );

  //list streams
  stream_list( p_stream_list, p_out_info );
}

//type for command table
typedef struct s_cmd_tab
{
  char * name; //command name
  char * shortname; //short command name
  char * params; //parameter format or NULL
  char * descr; //short description
  void (* func)( char * p_params, st_out_info * p_out_info ); //function to call
  char * description; //long description (including "  " at beginning and "\n" at end of every line) or NULL
} st_cmd_tab;

//command table
st_cmd_tab cmd_tab[] =
{
  {
    .name = "dyndest_list",
    .shortname = "ddl",
    .params = "<outport>",
    .descr = "list all dynamic destinations of an output port",
    .func = cmd_dyndest_list,
    .description = "  shows a list with all dynamic destinations that are currently\n"
                   "  receiving the active stream of the output port named <outport>\n"
                   "  \n"
                   "  e.g.: \"dyndest_list my_outport\"\n",
  },
  {
    .name = "dynsrc_add",
    .shortname = "dsa",
    .params = "<inport> <name> <addr-fam> <addr> <port> <protocol> <interval>",
    .descr = "add a dynamic source to an input port",
    .func = cmd_dynsrc_list_add,
    .description = "  adds a dynamic source called <name> to the input port called\n"
                   "  <inport>\n"
                   "  the dynamic stream is requested from <addr-fam> <addr>\n"
                   "  <port> using the dynamic extension of the protocol <protocol>\n"
                   "  with an request interval of <interval> seconds\n"
                   "  \n"
                   "  e.g.: \"dynsrc_add my_inport my_dynsrc in 192.168.0.1 2323 blp 10\"\n"
                   "  e.g.: \"dynsrc_add my_inport my_dynsrc_ipv6 in6 ::1 2323 mcuf 10\"\n",
  },
  {
    .name = "dynsrc_del",
    .shortname = "dsd",
    .params = "<inport> <name>",
    .descr = "delete a dynamic source from an input port",
    .func = cmd_dynsrc_list_del,
    .description = "  deletes the dynamic source called <name> of the input port called\n"
                   "  <inport>\n"
                   "  \n"
                   "  e.g.: \"dynsrc_del my_inport my_dynsrc\"\n",
  },
  {
    .name = "dynsrc_list",
    .shortname = "dsl",
    .params = "<inport>",
    .descr = "list all dynamic sources of an input port",
    .func = cmd_dynsrc_list,
    .description = "  shows a list with all dynamic sources of the input port called <inport>\n"
                   "  \n"
                   "  e.g.: \"dynsrc_list my_inport\"\n",
  },
  {
    .name = "end",
    .shortname = "e",
    .params = NULL,
    .descr = "end the session",
    .func = NULL,
    .description = NULL,
  },
  {
    .name = "exit",
    .shortname = "x",
    .params = NULL,
    .descr = "end the session",
    .func = NULL,
    .description = NULL,
  },
  {
    .name = "help",
    .shortname = "h",
    .params = "[<name>]",
    .descr = "show all commands / show description of command <name>",
    .func = cmd_help,
    .description = NULL,
  },
  {
    .name = "infilter_add",
    .shortname = "ifa",
    .params = "<inport> <index> <addr-fam> <net/mask> <port-port> {<protcol>|any} {<format>|-} <stream-name> [stop] [<outport-name> <priority> <timeout>]",
    .descr = "add an input filter rule to an input port",
    .func = cmd_infilter_list_add,
    .description = "  adds an input filter rule at the position <index> to the input filter\n"
                   "  list of the input port called <inport>\n"
                   "  if the incoming frame matches the address family <addr-fam>, the\n"
                   "  address range <net/mask>, the port range <port-port>, the protocol\n"
                   "  <proto> and the format <format> the stream is called <name> and is\n"
                   "  submitted to the output port <outport-name> with priority <priority>\n"
                   "  and timeout set to <timeout> seconds\n"
                   "  if [stop] is specified the processing of the list stops after this entry\n"
                   "  if [<outport-name> ...] is left out the frame is not submitted to an\n"
                   "  output port\n"
                   "  \n"
                   "  e.g.: \"infilter_add my_inport 0 in 12.34.56.0/24 2323 eblp - deny_stream stop\"\n"
                   "  e.g.: \"infilter_add my_inport 1 in6 0/0 1-65535 any 18x8-1/2 output_stream_ipv6 my_outport 40000 5\"\n"
                   "  e.g.: \"infilter_add my_inport 2 in 0.0.0.0/0 1-65535 any 18x8-1 output_stream my_outport 30000 5\"\n",
  },
  {
    .name = "infilter_del",
    .shortname = "ifd",
    .params = "<inport> <index>",
    .descr = "delete an input filter rule from an input port",
    .func = cmd_infilter_list_del,
    .description = "  deletes the input filter rule at position <index> of the input port\n"
                   "  called <inport>\n"
                   "  \n"
                   "  e.g.: \"infilter_del my_inport 2\"\n",
  },
  {
    .name = "infilter_list",
    .shortname = "ifl",
    .params = "<inport>",
    .descr = "list all input filter rules of an input port",
    .func = cmd_infilter_list,
    .description = "  shows a list with all input filter rules of the input port called\n"
                   "  <inport>\n"
                   "  \n"
                   "  e.g.: \"infilter_list my_inport\"\n",
  },
  {
    .name = "inport_add",
    .shortname = "ipa",
    .params = "<name> <addr-fam> <host> <port>",
    .descr = "add an input port",
    .func = cmd_inport_list_add,
    .description = "  creates a new input port called <name> at <addr-fam> <host> <port>\n"
                   "  \n"
                   "  e.g.: \"inport_add my_inport in 0.0.0.0 23231\"\n"
                   "  e.g.: \"inport_add my_inport_ipv6 in6 0 23231\"\n",
  },
  {
    .name = "inport_del",
    .shortname = "ipd",
    .params = "<name>",
    .descr = "delete an input port",
    .func = cmd_inport_list_del,
    .description = "  deletes the input port called <name>\n"
                   "  \n"
                   "  e.g.: \"inport_del my_inport\"\n",
  },
  {
    .name = "inport_list",
    .shortname = "ipl",
    .params = NULL,
    .descr = "list all input ports",
    .func = cmd_inport_list,
    .description = "  shows a list of all input ports\n"
                   "  \n"
                   "  e.g.: \"inport_list\"\n",
  },
  {
    .name = "outfilter_add",
    .shortname = "ofa",
    .params = "<outport> <index> <addr-fam> <net/mask> <port-port> {<protocol>|any} <dest-name> {deny|<timeout>} [<format>]",
    .descr = "add an output filter rule to an output port",
    .func = cmd_outfilter_list_add,
    .description = "  adds an output filter rule at the position <index> to the output filter\n"
                   "  list of the output port called <outport>\n"
                   "  if the incoming dynamic stream request matches the address family <addr-fam>,\n"
                   "  the address range <net/mask>, the port range <port-port> and the protocol\n"
                   "  <proto> the request / dynamic output stream is called <dest-name>\n"
                   "  if [deny] is specified the request is ignored\n"
                   "  if [<timeout>] [<format>] is specified a dynamic output stream is created with\n"
                   "  a timeout of <timeout> seconds which send in the format <format>\n"
                   "  \n"
                   "  e.g.: \"outfilter_add my_outport 0 in 12.34.56.0/24 2323 eblp deny_dest deny\"\n"
                   "  e.g.: \"outfilter_add my_outport 1 in6 0/0 1-65535 any dyn_dest_ipv6 30 18x8\"\n"
                   "  e.g.: \"outfilter_add my_outport 2 in 0.0.0.0/0 1-65535 any dyn_dest 26x20\"\n",
  },
  {
    .name = "outfilter_del",
    .shortname = "ofd",
    .params = "<outport> <index>",
    .descr = "delete an output filter rule from an output port",
    .func = cmd_outfilter_list_del,
    .description = "  deletes the output filter rule at position <index> of the output port\n"
                   "  called <outport>\n"
                   "  \n"
                   "  e.g.: \"outfilter_del my_outport 2\"\n",
  },
  {
    .name = "outfilter_list",
    .shortname = "ofl",
    .params = "<outport>",
    .descr = "list all output filter rules of an output port",
    .func = cmd_outfilter_list,
    .description = "  shows a list with all output filter rules of the output port called\n"
                   "  <outport>\n"
                   "  \n"
                   "  e.g.: \"outfilter_list my_outport\"\n",
  },
  {
    .name = "outport_add",
    .shortname = "opa",
    .params = "<name> <addr-fam> <host> <port> [<timeout>]",
    .descr = "add an output port",
    .func = cmd_outport_list_add,
    .description = "  creates a new output port called <name> at <addr-fam> <host> <port>\n"
                   "  with a timeout (repetition interval of last packet) of <timeout> ms\n"
                   "  \n"
                   "  e.g.: \"outport_add my_outport in 0.0.0.0 23230\"\n"
                   "  e.g.: \"outport_add my_outport_ipv6 in6 0 23230\"\n",
  },
  {
    .name = "outport_del",
    .shortname = "opd",
    .params = "<name>",
    .descr = "delete an output port",
    .func = cmd_outport_list_del,
    .description = "  deletes the output port called <name>\n"
                   "  \n"
                   "  e.g.: \"outport_del my_outport\"\n",
  },
  {
    .name = "outport_list",
    .shortname = "opl",
    .params = NULL,
    .descr = "list all output ports",
    .func = cmd_outport_list,
    .description = "  shows a list of all output ports\n"
                   "  \n"
                   "  e.g.: \"outport_list\"\n",
  },
  {
    .name = "quit",
    .shortname = "q",
    .params = NULL,
    .descr = "end the session",
    .func = NULL,
    .description = NULL,
  },
  {
    .name = "statdest_add",
    .shortname = "sda",
    .params = "<outport> <name> <addr-fam> <addr> <port> <protocol> [<format>]",
    .descr = "add a static destination to an output port",
    .func = cmd_statdest_list_add,
    .description = "  adds a static destination called <name> to the output port called\n"
                   "  <outport>\n"
                   "  the current stream is sent to <addr-fam> <addr> <port> using the\n"
                   "  protocol <protocol> int the format <format>\n"
                   "  \n"
                   "  e.g.: \"statdest_add my_outport my_statdest in 192.168.0.1 2342 blp 18x8-1/2\"\n"
                   "  e.g.: \"statdest_add my_outport my_statdest_ipv6 in6 ::1 2342 mcuf 26x20-1\"\n",
  },
  {
    .name = "statdest_del",
    .shortname = "sdd",
    .params = "<outport> <name>",
    .descr = "delete a static destination from an output port",
    .func = cmd_statdest_list_del,
    .description = "  deletes the static destination called <name> of the output port called\n"
                   "  <outport>\n"
                   "  \n"
                   "  e.g.: \"statdest_del my_outport my_statdest\"\n",
  },
  {
    .name = "statdest_format",
    .shortname = "sdf",
    .params = "<outport> <name> [<format>]",
    .descr = "set new picture format for a static destination of an output port",
    .func = cmd_statdest_list_format,
    .description = "  sets / changes the format of the static destination called <name> of\n"
                   "  the output port called <outport>\n"
                   "  \n"
                   "  e.g.: \"statdest_format my_outport my_statdest 26x20-1/256\"\n",
  },
  {
    .name = "statdest_list",
    .shortname = "sdl",
    .params = "<outport>",
    .descr = "list all static destinations of an output port",
    .func = cmd_statdest_list,
    .description = "  shows a list with all static destination sources of the output port called\n"
                   "  <outport>\n"
                   "  \n"
                   "  e.g.: \"statdest_list my_outport\"\n",
  },
  {
    .name = "stream_list",
    .shortname = "stl",
    .params = "<outport>",
    .descr = "list all streams of an output port",
    .func = cmd_stream_list,
    .description = "  shows a list with all streams that are currently arriving at the output port\n"
                   "  called <outport>\n"
                   "  the current stream is marked with \"<<<=====\"\n"
                   "  \n"
                   "  e.g.: \"stream_list my_outport\"\n",
  },
};

//help command
void cmd_help( char * p_params, st_out_info * p_out_info )
{
  char * p_cmd;
  unsigned int i;

  //get command
  tools_str_first_words( p_params, 1, &p_cmd, &p_params );

  //command found in parameters
  if( p_cmd[0] != 0 )
  {
    //search for command
    for( i = 0; i < count( cmd_tab ); i++ )
      if( strcasecmp( p_cmd, cmd_tab[i].name ) == 0
          || (cmd_tab[i].shortname != NULL
              && strcasecmp( p_cmd, cmd_tab[i].shortname ) == 0) )
        break;
    //command found in table
    if( i < count( cmd_tab ) )
    {
      //print command in long and short form
      out_str( p_out_info, "  " );
      out_str( p_out_info, cmd_tab[i].name );
      if( cmd_tab[i].params != NULL )
      {
        out_str( p_out_info, " " );
        out_str( p_out_info, cmd_tab[i].params );
      }
      out_str( p_out_info, "\n" );
      if( cmd_tab[i].shortname != NULL )
      {
        out_str( p_out_info, "  " );
        out_str( p_out_info, cmd_tab[i].shortname );
        if( cmd_tab[i].params != NULL )
        {
          out_str( p_out_info, " " );
          out_str( p_out_info, cmd_tab[i].params );
        }
        out_str( p_out_info, "\n" );
      }
      out_str( p_out_info, "\n  " );
      out_str( p_out_info, cmd_tab[i].descr );
      out_str( p_out_info, "\n" );
      //print long description
      if( cmd_tab[i].description != NULL )
      {
        out_str( p_out_info, "\n" );
        out_str( p_out_info, cmd_tab[i].description );
      }
    }
    //command not found in table
    else
    {
      //print error message
      out_str( p_out_info, "  no help found to \"" );
      out_str( p_out_info, p_cmd );
      out_str( p_out_info, "\"\n" );
    }
    return;
  }

  //print command table and short descriptions
  for( i = 0; i < count( cmd_tab ); i++ )
  {
    out_str( p_out_info, "  " );
    out_str( p_out_info, cmd_tab[i].name );
    if( cmd_tab[i].shortname != NULL )
    {
      out_str( p_out_info, " / " );
      out_str( p_out_info, cmd_tab[i].shortname );
    }
    if( cmd_tab[i].params != NULL )
    {
      out_str( p_out_info, " " );
      out_str( p_out_info, cmd_tab[i].params );
    }
    out_str( p_out_info, "\n    " );
    out_str( p_out_info, cmd_tab[i].descr );
    out_str( p_out_info, "\n" );
  }
  out_str( p_out_info, "\n  type \"help <command>\" for additional info\n" );
}

//process a command
//returns 1 if end was requested, returns 0 otherwise
int cmd_exec( char * p_line, st_out_info * p_out_info )
{
  char * p_chr, * p_cmd, * p_params;
  unsigned int i;

  //truncate command at first hash (for comments) or newline
  p_chr = strchr( p_line, '#' );
  if( p_chr != NULL )
    *p_chr = 0;
  p_chr = strchr( p_line, '\r' );
  if( p_chr != NULL )
    *p_chr = 0;
  p_chr = strchr( p_line, '\n' );
  if( p_chr != NULL )
    *p_chr = 0;

  //ignore empty lines
  if( p_line[0] == 0 )
    return 0;

  //print command
  out_str( p_out_info, "\n" );
  out_str( p_out_info, p_line );
  out_str( p_out_info, "\n" );

  //get command (first word of line)
  tools_str_first_words( p_line, 1, &p_cmd, &p_params );
  //ignore empty command
  if( p_cmd[0] == 0 )
    return 0;

  //find command in command table
  for( i = 0; i < count( cmd_tab ); i++ )
    if( strcasecmp( p_cmd, cmd_tab[i].name ) == 0
        || (cmd_tab[i].shortname != NULL
            && strcasecmp( p_cmd, cmd_tab[i].shortname ) == 0) )
      break;
  //command found
  if( i < count( cmd_tab ) )
  {
    //return 1 to indicate end if no function
    if( cmd_tab[i].func == NULL )
      return 1;
    //call command
    cmd_tab[i].func( p_params, p_out_info );
    out_str( p_out_info, "\n" );
  }
  //command not found
  else
    out_str( p_out_info, "  unknown command - use \"help\"\n\n" );

  return 0;
}
