/* BLINKENLIGHTS PROXY
 * version 0.100 date 2008-03-21
 * Copyright (C) 2003-2006 Stefan Schuermans <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <netdb.h>

#include "tools.h"
#include "addr.h"

//read address from string
//address consists of 3 strings <address family> <host> <port>
//returns 0 on success, a negative value on error
int addr_str2addr( char * p_str, st_addr * p_addr, st_out_info * p_out_info )
{
  char * p_af, * p_host, * p_port, * p_chr;
  struct hostent * p_hostent;
  unsigned long value;
  in_port_t port;

  //get address family, host and port
  tools_str_first_words( p_str, 1, &p_af, &p_str );
  tools_str_first_words( p_str, 1, &p_host, &p_str );
  tools_str_first_words( p_str, 1, &p_port, &p_str );

  //IPv4
  if( strcasecmp( p_af, "in" ) == 0 || strcasecmp( p_af, "in4" ) == 0 )
  {
    p_addr->af = AF_INET;
    p_addr->sa_len = sizeof( p_addr->sa.sin );
    p_addr->sa.sin.sin_family = AF_INET;

    //get IPv4-address from host
    if( ! inet_pton( AF_INET, p_host, &p_addr->sa.sin.sin_addr ) )
    {
      //could not parse IPv4-address, perhaps it is a hostname
      //try to resolve name
      p_hostent = gethostbyname2( p_host, AF_INET );
      if( p_hostent == NULL || p_hostent->h_addr_list == NULL || p_hostent->h_addr_list[0] == NULL )
      {
        out_str( p_out_info, "  error: cannot resolve (IPv4) \"" );
        out_str( p_out_info, p_host );
        out_str( p_out_info, "\"\n" );
        return -ENOENT;
      }
      p_addr->sa.sin.sin_addr = *(struct in_addr *)(p_hostent->h_addr_list[0]);
    }

    //get port
    value = strtoul( p_port, &p_chr, 0 );
    port = (in_port_t)value;
    if( *p_port == 0 || *p_chr != 0 || port != value )
    {
      out_str( p_out_info, "  error: invalid port number \"" );
      out_str( p_out_info, p_port );
      out_str( p_out_info, "\"\n" );
      return -EINVAL;
    }
    p_addr->sa.sin.sin_port = htons( port );

    //success
    return 0;
  } //IPv4

  //IPv6
  else if( strcasecmp( p_af, "in6" ) == 0 )
  {
    p_addr->af = AF_INET6;
    p_addr->sa_len = sizeof( p_addr->sa.sin6 );
    p_addr->sa.sin6.sin6_family = AF_INET6;

    //get IPv6-address from host
    if( ! inet_pton( AF_INET6, p_host, &p_addr->sa.sin6.sin6_addr ) )
    {
      //could not parse IPv6-address, perhaps it is a hostname
      //try to resolve name
      p_hostent = gethostbyname2( p_host, AF_INET6 );
      if( p_hostent == NULL || p_hostent->h_addr_list == NULL || p_hostent->h_addr_list[0] == NULL )
      {
        out_str( p_out_info, "  error: cannot resolve (IPv6) \"" );
        out_str( p_out_info, p_host );
        out_str( p_out_info, "\"\n" );
        return -ENOENT;
      }
      p_addr->sa.sin6.sin6_addr = *(struct in6_addr *)(p_hostent->h_addr_list[0]);
    }

    //get port
    value = strtoul( p_port, &p_chr, 0 );
    port = (in_port_t)value;
    if( *p_port == 0 || *p_chr != 0 || port != value )
    {
      out_str( p_out_info, "  error: invalid port number \"" );
      out_str( p_out_info, p_port );
      out_str( p_out_info, "\"\n" );
      return -EINVAL;
    }
    p_addr->sa.sin6.sin6_port = htons( port );

    //success
    return 0;
  } //IPv6

  //unknown address family
  else
  {
    out_str( p_out_info, "  error: unknown address family \"" );
    out_str( p_out_info, p_af );
    out_str( p_out_info, "\" (try \"in\" or \"in6\")\n" );
    return -EINVAL;
  }
}

//set up addr structure from sa and sa_len members
//returns 0 on success, a negative value on error
int addr_sa2addr( st_addr * p_addr )
{
  //get address family from sockaddr structure
  p_addr->af = p_addr->sa.sa.sa_family;

  //switch by address family
  switch( p_addr->af )
  {
    //IPv4
    case AF_INET:
      if( p_addr->sa_len != sizeof( p_addr->sa.sin ) )
        return -EINVAL;
      return 0;

    //IPv6
    case AF_INET6:
      if( p_addr->sa_len != sizeof( p_addr->sa.sin6 ) )
        return -EINVAL;
      return 0;

    //unknown address family
    default:
      return -EINVAL;
  } //switch( p_addr->af )
}

//get string from address
//address consists of 3 strings <address family> <host> <port>
//returns 0 on success, a negative value on error
int addr_addr2str( st_addr * p_addr, char * p_str, unsigned int str_buf_size, st_out_info * p_out_info )
{
  //switch by address family
  switch( p_addr->af )
  {
    //IPv4
    case AF_INET:
      //return string
      if( str_buf_size < 3 + INET_ADDRSTRLEN + 6 + 1 )
      {
        out_str( p_out_info, "  error: cannot convert address to string (buffer too small)\n" );
        return -ENOMEM;
      }
      strcpy( p_str, "in " );
      p_str += strlen( p_str );
      inet_ntop( AF_INET, &p_addr->sa.sin.sin_addr, p_str, INET_ADDRSTRLEN );
      p_str += strlen( p_str );
      sprintf( p_str, " %u", ntohs( p_addr->sa.sin.sin_port ) );
      return 0;

    //IPv6
    case AF_INET6:
      //return string
      if( str_buf_size < 4 + INET6_ADDRSTRLEN + 6 + 1 )
      {
        out_str( p_out_info, "  error: cannot convert address to string (buffer too small)\n" );
        return -ENOMEM;
      }
      strcpy( p_str, "in6 " );
      p_str += strlen( p_str );
      inet_ntop( AF_INET6, &p_addr->sa.sin6.sin6_addr, p_str, INET6_ADDRSTRLEN );
      p_str += strlen( p_str );
      sprintf( p_str, " %u", ntohs( p_addr->sa.sin6.sin6_port ) );
      return 0;

    //unknown address family
    default:
      out_str( p_out_info, "  error: unknown address family\n" );
      return -EINVAL;
  } //switch( p_addr->af )
}

//output address
//address consists of 3 strings <address family> <host> <port>
void addr_addr_out( st_addr * p_addr, st_out_info * p_out_info )
{
  char str[max( 3 + INET_ADDRSTRLEN + 6 + 1, 4 + INET6_ADDRSTRLEN + 6 + 1 )];
  char * p_str;

  //switch by address family
  switch( p_addr->af )
  {
    //IPv4
    case AF_INET:
      //output string
      p_str = str;
      strcpy( p_str, "in " );
      p_str += strlen( p_str );
      inet_ntop( AF_INET, &p_addr->sa.sin.sin_addr, p_str, INET_ADDRSTRLEN );
      p_str += strlen( p_str );
      sprintf( p_str, " %u", ntohs( p_addr->sa.sin.sin_port ) );
      out_str( p_out_info, str );
      break;

    //IPv6
    case AF_INET6:
      //output string
      p_str = str;
      strcpy( p_str, "in6 " );
      p_str += strlen( p_str );
      inet_ntop( AF_INET6, &p_addr->sa.sin6.sin6_addr, p_str, INET6_ADDRSTRLEN );
      p_str += strlen( p_str );
      sprintf( p_str, " %u", ntohs( p_addr->sa.sin6.sin6_port ) );
      out_str( p_out_info, str );
      break;

    //unknown address family
    default:
      out_str( p_out_info, "<unknown address family>" );
  } //switch( p_addr->af )
}

//compare addresses (e.g. used to sort lists)
int addr_cmp( st_addr * p_addr1, st_addr * p_addr2 )
{
  int cmp;

  //compare address families
  if( p_addr1->af < p_addr2->af )
    return -1;
  if( p_addr1->af > p_addr2->af )
    return 1;

  //switch by address family
  switch( p_addr1->af )
  {
    //IPv4
    case AF_INET:
      //compare IPs
      cmp = memcmp( &p_addr1->sa.sin.sin_addr, &p_addr2->sa.sin.sin_addr, sizeof( p_addr1->sa.sin.sin_addr ) );
      if( cmp != 0 )
        return cmp;
      //compare ports
      cmp = ntohs( p_addr1->sa.sin.sin_port ) - ntohs( p_addr2->sa.sin.sin_port );
      if( cmp < 0 )
        return -1;
      if( cmp > 0 )
        return 1;
      return 0;

    //IPv6
    case AF_INET6:
      //compare IPs
      cmp = memcmp( &p_addr1->sa.sin6.sin6_addr, &p_addr2->sa.sin6.sin6_addr, sizeof( p_addr1->sa.sin6.sin6_addr ) );
      if( cmp != 0 )
        return cmp;
      //compare ports
      cmp = ntohs( p_addr1->sa.sin6.sin6_port ) - ntohs( p_addr2->sa.sin6.sin6_port );
      if( cmp < 0 )
        return -1;
      if( cmp > 0 )
        return 1;
      return 0;

    //unknown address family
    default:
      //we cannot compare addresses of unknown address families
      //so we return 0 - this will result in unsorted addresses of unknown address families
      return 0;
  } //switch( p_addr->af )
}

//read address range from string
//address range consists of 3 strings <address family> <net/mask> <port-port>
//returns 0 on success, a negative value on error
int addr_str2range( char * p_str, st_addr_range * p_range, st_out_info * p_out_info )
{
  char * p_af, * p_netrange, * p_ports, * p_chr;
  char * p_net, * p_mask, * p_port_min, * p_port_max;
  struct hostent * p_hostent;
  unsigned long value;
  in_port_t port_min, port_max, port_tmp;

  //get address family, network range and ports
  tools_str_first_words( p_str, 1, &p_af, &p_str );
  tools_str_first_words( p_str, 1, &p_netrange, &p_str );
  tools_str_first_words( p_str, 1, &p_ports, &p_str );

  //check network range
  if( *p_netrange == 0 )
  {
    out_str( p_out_info, "  error: empty network range is invalid\n" );
    return -EINVAL;
  }
  //split network range into network and mask
  p_chr = strchr( p_netrange, '/' );
  p_net = p_netrange;
  p_mask = "";
  if( p_chr != NULL )
  {
    *p_chr = 0;
    p_mask = p_chr + 1;
  }
  
  //check port range
  if( *p_ports == 0 )
  {
    out_str( p_out_info, "  error: empty port range is invalid\n" );
    return -EINVAL;
  }
  //split port range into minimum and maximum port
  p_chr = strchr( p_ports, '-' );
  p_port_min = p_ports;
  p_port_max = p_ports;
  if( p_chr != NULL )
  {
    *p_chr = 0;
    p_port_max = p_chr + 1;
  }

  //IPv4
  if( strcasecmp( p_af, "in" ) == 0 || strcasecmp( p_af, "in4" ) == 0 )
  {
    p_range->af = AF_INET;

    //get IPv4-address from net
    if( ! inet_pton( AF_INET, p_net, &p_range->range.in.net ) )
    {
      //could not parse IPv4-address, perhaps it is a hostname
      //try to resolve name
      p_hostent = gethostbyname2( p_net, AF_INET );
      if( p_hostent == NULL || p_hostent->h_addr_list == NULL || p_hostent->h_addr_list[0] == NULL )
      {
        out_str( p_out_info, "  error: cannot resolve (IPv4) \"" );
        out_str( p_out_info, p_net );
        out_str( p_out_info, "\"\n" );
        return -ENOENT;
      }
      p_range->range.in.net = *(struct in_addr *)(p_hostent->h_addr_list[0]);
    }
    //get IPv4-address from mask
    if( *p_mask == 0 )
      //no mask
      p_range->range.in.mask = (struct in_addr){ .s_addr = 0xFFFFFFFF };
    else
    {
      value = strtoul( p_mask, &p_chr, 0 );
      if( *p_chr == 0 && value <= 32 )
      {
        //"/<number>" mask
        p_range->range.in.mask = (struct in_addr){ .s_addr = htonl( value <= 0 ? 0 : 0xFFFFFFFF << (32 - value) ) };
      }
      else if( ! inet_pton( AF_INET, p_mask, &p_range->range.in.mask ) ) //real mask
      {
        //error
        out_str( p_out_info, "  error: invalid mask (IPv4) \"" );
        out_str( p_out_info, p_mask );
        out_str( p_out_info, "\"\n" );
        return -EINVAL;
      }
    }

    //get minimum port
    if( *p_port_min == 0 )
      port_min = 1;
    else
    {
      value = strtoul( p_port_min, &p_chr, 0 );
      port_min = (in_port_t)value;
      if( *p_chr != 0 || port_min != value )
      {
        out_str( p_out_info, "  error: invalid port number \"" );
        out_str( p_out_info, p_port_min );
        out_str( p_out_info, "\"\n" );
        return -EINVAL;
      }
    }
    //get maximum port
    if( *p_port_max == 0 )
      port_max = 65535;
    else
    {
      value = strtoul( p_port_max, &p_chr, 0 );
      port_max = (in_port_t)value;
      if( *p_chr != 0 || port_max != value )
      {
        out_str( p_out_info, "  error: invalid port number \"" );
        out_str( p_out_info, p_port_max );
        out_str( p_out_info, "\"\n" );
        return -EINVAL;
      }
    }
    //check minimum port
    if( port_min == 0 )
      port_min = 1;
    if( port_min > port_max )
    {
      port_tmp = port_min;
      port_min = port_max;
      port_max = port_tmp;
    }
    //store port range
    p_range->range.in.port_min = htons( port_min );
    p_range->range.in.port_max = htons( port_max );

    //success
    return 0;
  } //IPv4

  //IPv6
  else if( strcasecmp( p_af, "in6" ) == 0 )
  {
    p_range->af = AF_INET6;

    //get IPv6-address from net
    if( ! inet_pton( AF_INET6, p_net, &p_range->range.in6.net ) )
    {
      //could not parse IPv6-address, perhaps it is a hostname
      //try to resolve name
      p_hostent = gethostbyname2( p_net, AF_INET6 );
      if( p_hostent == NULL || p_hostent->h_addr_list == NULL || p_hostent->h_addr_list[0] == NULL )
      {
        out_str( p_out_info, "  error: cannot resolve (IPv6) \"" );
        out_str( p_out_info, p_net );
        out_str( p_out_info, "\"\n" );
        return -ENOENT;
      }
      p_range->range.in6.net = *(struct in6_addr *)(p_hostent->h_addr_list[0]);
    }
    //get IPv6-address from mask
    if( *p_mask == 0 )
      //no mask
      p_range->range.in6.mask = (struct in6_addr){ .in6_u = { .u6_addr32 = { 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF } } };
    else
    {
      value = strtoul( p_mask, &p_chr, 0 );
      if( *p_chr == 0 && value <= 128 )
      {
        //"/<number>" mask
        p_range->range.in6.mask = (struct in6_addr){ .in6_u = { .u6_addr32 = {
          htonl( value <= 0 ? 0 : 0xFFFFFFFF << (value <= 32 ? 32 - value : 0) ),
          htonl( value <= 32 ? 0 : 0xFFFFFFFF << (value <= 64 ? 64 - value : 0) ),
          htonl( value <= 64 ? 0 : 0xFFFFFFFF << (value <= 96 ? 96 - value : 0) ),
          htonl( value <= 96 ? 0 : 0xFFFFFFFF << (32 - value) )
        } } };
      }
      else if( ! inet_pton( AF_INET6, p_mask, &p_range->range.in6.mask ) ) //real mask
      {
        //error
        out_str( p_out_info, "  error: invalid mask (IPv6) \"" );
        out_str( p_out_info, p_mask );
        out_str( p_out_info, "\"\n" );
        return -EINVAL;
      }
    }

    //get minimum port
    if( *p_port_min == 0 )
      port_min = 1;
    else
    {
      value = strtoul( p_port_min, &p_chr, 0 );
      port_min = (in_port_t)value;
      if( *p_chr != 0 || port_min != value )
      {
        out_str( p_out_info, "  error: invalid port number \"" );
        out_str( p_out_info, p_port_min );
        out_str( p_out_info, "\"\n" );
        return -EINVAL;
      }
    }
    //get maximum port
    if( *p_port_max == 0 )
      port_max = 65535;
    else
    {
      value = strtoul( p_port_max, &p_chr, 0 );
      port_max = (in_port_t)value;
      if( *p_chr != 0 || port_max != value )
      {
        out_str( p_out_info, "  error: invalid port number \"" );
        out_str( p_out_info, p_port_max );
        out_str( p_out_info, "\"\n" );
        return -EINVAL;
      }
    }
    //check minimum port
    if( port_min < 1 )
      port_min = 1;
    if( port_min > port_max )
    {
      port_tmp = port_min;
      port_min = port_max;
      port_max = port_tmp;
    }
    //store port range
    p_range->range.in6.port_min = htons( port_min );
    p_range->range.in6.port_max = htons( port_max );

    //success
    return 0;
  } //IPv6

  //unknown address family
  else
  {
    out_str( p_out_info, "  error: unknown address family \"" );
    out_str( p_out_info, p_af );
    out_str( p_out_info, "\" (try \"in\" or \"in6\")\n" );
    return -EINVAL;
  }
}

//get string from address range
//address range consists of 3 strings <address family> <net/mask> <port-port>
//returns 0 on success, a negative value on error
int addr_range2str( st_addr_range * p_range, char * p_str, unsigned int str_buf_size, st_out_info * p_out_info )
{
  //switch by address family
  switch( p_range->af )
  {
    //IPv4
    case AF_INET:
      //return string
      if( str_buf_size < 3 + INET_ADDRSTRLEN + 1 + INET_ADDRSTRLEN + 12 + 1 )
      {
        out_str( p_out_info, "  error: cannot convert address range to string (buffer too small)\n" );
        return -ENOMEM;
      }
      strcpy( p_str, "in " );
      p_str += strlen( p_str );
      inet_ntop( AF_INET, &p_range->range.in.net, p_str, INET_ADDRSTRLEN );
      p_str += strlen( p_str );
      strcpy( p_str, "/" );
      p_str += strlen( p_str );
      inet_ntop( AF_INET, &p_range->range.in.mask, p_str, INET_ADDRSTRLEN );
      p_str += strlen( p_str );
      sprintf( p_str, " %u-%u", ntohs( p_range->range.in.port_min ), ntohs( p_range->range.in.port_max ) );
      return 0;

    //IPv6
    case AF_INET6:
      //return string
      if( str_buf_size < 4 + INET6_ADDRSTRLEN + 1 + INET6_ADDRSTRLEN + 12 + 1 )
      {
        out_str( p_out_info, "  error: cannot convert address to string (buffer too small)\n" );
        return -ENOMEM;
      }
      strcpy( p_str, "in6 " );
      p_str += strlen( p_str );
      inet_ntop( AF_INET6, &p_range->range.in6.net, p_str, INET6_ADDRSTRLEN );
      p_str += strlen( p_str );
      strcpy( p_str, "/" );
      p_str += strlen( p_str );
      inet_ntop( AF_INET6, &p_range->range.in6.mask, p_str, INET6_ADDRSTRLEN );
      p_str += strlen( p_str );
      sprintf( p_str, " %u-%u", ntohs( p_range->range.in6.port_min ), ntohs( p_range->range.in6.port_max ) );
      return 0;

    //unknown address family
    default:
      out_str( p_out_info, "  error: unknown address family\n" );
      return -EINVAL;
  } //switch( p_range->af )
}

//output address range
//address range consists of 3 strings <address family> <net/mask> <port-port>
void addr_range_out( st_addr_range * p_range, st_out_info * p_out_info )
{
  char str[max( 3 + INET_ADDRSTRLEN + 1 + INET_ADDRSTRLEN + 12 + 1, 4 + INET6_ADDRSTRLEN + 1 + INET6_ADDRSTRLEN + 12 + 1 )];
  char * p_str;

  //switch by address family
  switch( p_range->af )
  {
    //IPv4
    case AF_INET:
      //output string
      p_str = str;
      strcpy( p_str, "in " );
      p_str += strlen( p_str );
      inet_ntop( AF_INET, &p_range->range.in.net, p_str, INET_ADDRSTRLEN );
      p_str += strlen( p_str );
      strcpy( p_str, "/" );
      p_str += strlen( p_str );
      inet_ntop( AF_INET, &p_range->range.in.mask, p_str, INET_ADDRSTRLEN );
      p_str += strlen( p_str );
      sprintf( p_str, " %u-%u", ntohs( p_range->range.in.port_min ), ntohs( p_range->range.in.port_max ) );
      out_str( p_out_info, str );
      break;

    //IPv6
    case AF_INET6:
      //output string
      p_str = str;
      strcpy( p_str, "in6 " );
      p_str += strlen( p_str );
      inet_ntop( AF_INET6, &p_range->range.in6.net, p_str, INET6_ADDRSTRLEN );
      p_str += strlen( p_str );
      strcpy( p_str, "/" );
      p_str += strlen( p_str );
      inet_ntop( AF_INET6, &p_range->range.in6.mask, p_str, INET6_ADDRSTRLEN );
      p_str += strlen( p_str );
      sprintf( p_str, " %u-%u", ntohs( p_range->range.in6.port_min ), ntohs( p_range->range.in6.port_max ) );
      out_str( p_out_info, str );
      break;

    //unknown address family
    default:
      out_str( p_out_info, "<unknown address family>" );
  } //switch( p_range->af )
}

//check if an addr matches an address range
//retuns 1 on match, 0 on mismatch
int addr_match( st_addr * p_addr, st_addr_range * p_range )
{
  int i;
  uint32_t addr, mask, net;
  in_port_t port, port_min, port_max;

  //check address families
  if( p_addr->af != p_range->af )
    return 0;

  //switch by address family
  switch( p_addr->af )
  {
    //IPv4
    case AF_INET:
      //check host
      addr = ntohl( p_addr->sa.sin.sin_addr.s_addr );
      net = ntohl( p_range->range.in.net.s_addr );
      mask = ntohl( p_range->range.in.mask.s_addr );
      if( (addr & mask) != (net & mask) )
        return 0;
      //check port
      port = ntohs( p_addr->sa.sin.sin_port );
      port_min = ntohs( p_range->range.in.port_min );
      port_max = ntohs( p_range->range.in.port_max );
      if( port < port_min || port > port_max )
        return 0;
      //match
      return 1;

    //IPv6
    case AF_INET6:
      //check host
      for( i = 0; i < 4; i++ )
      {
        addr = ntohs( p_addr->sa.sin6.sin6_addr.in6_u.u6_addr32[i] );
        net = ntohs( p_range->range.in6.net.in6_u.u6_addr32[i] );
        mask = ntohs( p_range->range.in6.mask.in6_u.u6_addr32[i] );
        if( (addr & mask) != (net & mask) )
          return 0;
      }
      //check port
      port = ntohl( p_addr->sa.sin6.sin6_port );
      port_min = ntohl( p_range->range.in6.port_min );
      port_max = ntohl( p_range->range.in6.port_max );
      if( port < port_min || port > port_max )
        return 0;
      //match
      return 1;

    //unknown address family
    default:
      return 0;
  } //switch( p_addr->af )
}
