// BlinkenStreamView
// version 0.1.2 date 2014-01-18
// Copyright 2005-2014 Stefan Schuermans <stefan@schuermans.info>
// Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
// a blinkenarea.org project
// powered by eventphone.de

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <gtk/gtk.h>

#include "callbacks.h"
#include "net.h"
#include "support.h"

// import window widget variables
extern GtkWidget *Config;
extern GtkWidget *Display;

// current configuration
struct sockaddr_in net_listen;
struct sockaddr_in net_send;
unsigned int net_protocol;
unsigned int net_zoom;
int net_sock;
GIOChannel *net_io_channel;
guint net_io_watch;

// last frame
unsigned int net_frame_height = 0;
unsigned int net_frame_width = 0;
unsigned int net_frame_channels = 0;
unsigned int net_frame_maxval = 0;
unsigned char net_frame_data[65536];

// send a close
void net_send_close()
{
  if (net_sock == -1 || net_send.sin_addr.s_addr == INADDR_NONE)
    return;

  switch (net_protocol) {
  case net_protocol_BLP:
    sendto(net_sock, "\xDE" "\xAD" "\xBE" "\xCD" "CLOSE", 9, 0,
           (struct sockaddr *)&net_send, sizeof(net_send));
    break;
  case net_protocol_EBLP:
    sendto(net_sock, "\xDE" "\xAD" "\xBE" "\xCD" "CLOSE256", 12, 0,
           (struct sockaddr *)&net_send, sizeof(net_send));
    break;
  case net_protocol_MCUF:
    sendto(net_sock, "\x42" "\x42" "\x42" "\x43" "\0\0\0\0\0\0\0\0", 12, 0,
           (struct sockaddr *)&net_send, sizeof(net_send));
    break;
  }
}

// send a request
void net_send_request()
{
  if (net_sock == -1 || net_send.sin_addr.s_addr == INADDR_NONE)
    return;

  switch (net_protocol) {
  case net_protocol_BLP:
    sendto(net_sock, "\xDE" "\xAD" "\xBE" "\xCD" "REFRESH", 11, 0,
           (struct sockaddr *)&net_send, sizeof(net_send));
    break;
  case net_protocol_EBLP:
    sendto(net_sock, "\xDE" "\xAD" "\xBE" "\xCD" "REFRESH256", 14, 0,
           (struct sockaddr *)&net_send, sizeof(net_send));
    break;
  case net_protocol_MCUF:
    sendto(net_sock, "\x42" "\x42" "\x42" "\x42" "\0\0\0\0\0\0\0\0", 12, 0,
           (struct sockaddr *)&net_send, sizeof(net_send));
    break;
  }
}

// show reception
static void net_show_reception(char *text)
{
  GtkEditable *p_reception;
  gint pos;

  p_reception =
      (GtkEditable *) lookup_widget((GtkWidget *) Config, "Reception");
  gtk_editable_delete_text(p_reception, 0, -1);
  pos = 0;
  gtk_editable_insert_text(p_reception, text, strlen(text), &pos);
}

// redraw last frame
static void net_redraw()
{
  GtkImage *p_image;
  gint width, height;
  unsigned int z, y, x, si, di;
  unsigned char *p_rgb_data, r, g, b;
  GdkPixmap *p_pixmap;
  GdkGC *p_gc;

  p_image = (GtkImage *) lookup_widget((GtkWidget *) Display, "Image");

  // no frame available
  if (net_frame_height == 0 || net_frame_width == 0 || net_frame_channels == 0
      || net_frame_maxval == 0) {
    // clear image
    gtk_image_set_from_stock(p_image, 0, GTK_ICON_SIZE_INVALID);
    return;
  }
  // get size of image
  width = net_zoom * net_frame_width;
  height = net_zoom * net_frame_height;

  // convert frame to rgb data
  p_rgb_data = malloc(height * width * 3);
  if (p_rgb_data == NULL)
    return;
  si = 0;
  di = 0;
  for (y = 0; y < net_frame_height; y++) {
    for (x = 0; x < net_frame_width; x++) {
      r = net_frame_data[si++];
      if (net_frame_channels >= 2) {
        g = net_frame_data[si++];
        if (net_frame_channels >= 3) {
          b = net_frame_data[si++];
          if (net_frame_channels >= 3)
            si += net_frame_channels - 3;
        } else
          b = 0;
      } else {
        g = r;
        b = r;
      }
      if (net_frame_maxval < 255) {
        r = (unsigned char)(((unsigned int)r * 255 + net_frame_maxval / 2) /
                            net_frame_maxval);
        g = (unsigned char)(((unsigned int)g * 255 + net_frame_maxval / 2) /
                            net_frame_maxval);
        b = (unsigned char)(((unsigned int)b * 255 + net_frame_maxval / 2) /
                            net_frame_maxval);
      }
      for (z = 0; z < net_zoom; z++) {
        p_rgb_data[di++] = r;
        p_rgb_data[di++] = g;
        p_rgb_data[di++] = b;
      }
    }
    for (z = 1; z < net_zoom; z++) {
      memcpy(&p_rgb_data[di], &p_rgb_data[di - width * 3], width * 3);
      di += width * 3;
    }
  }

  // draw rgb data to pixmap
  p_pixmap = gdk_pixmap_new(NULL, width, height, 24);
  if (p_pixmap == NULL) {
    free(p_rgb_data);
    return;
  }
  p_gc =
      gdk_gc_new((GdkDrawable *)
                 gtk_widget_get_parent_window((GtkWidget *) p_image));
  if (p_gc == NULL) {
    free(p_rgb_data);
    g_object_unref(p_pixmap);
    return;
  }
  gdk_draw_rgb_image(p_pixmap, p_gc, 0, 0, width, height, GDK_RGB_DITHER_NONE,
                     p_rgb_data, width * 3);
  free(p_rgb_data);
  g_object_unref(p_gc);

  // put pixmap into image
  gtk_image_set_from_pixmap(p_image, p_pixmap, NULL);
  g_object_unref(p_pixmap);
}

// incoming packet
gboolean net_packet(GIOChannel * source, GIOCondition condition,
                    gpointer data)
{
  unsigned char buffer[65536];
  struct sockaddr_in addr;
  socklen_t addr_len;
  int len;
  char *proto, text[256];
  unsigned int height, width, channels, maxval;
  time_t t;
  struct tm tm;

  if (net_sock == -1)
    return FALSE;

  addr_len = sizeof(addr);
  len =
      recvfrom(net_sock, buffer, sizeof(buffer), 0, (struct sockaddr *)&addr,
               &addr_len);
  if (len <= 0)
    return FALSE;

  // too short
  if (len <= 12) {
    proto = "too short";
    height = 0;
    width = 0;
    channels = 0;
    maxval = 1;
  }
  // BLP
  else if (memcmp(buffer, "\xDE" "\xAD" "\xBE" "\xEF", 4) == 0) {
    proto = "BLP";
    height = ntohs(*(uint16_t *) (buffer + 10));
    width = ntohs(*(uint16_t *) (buffer + 8));
    channels = 1;
    maxval = 1;
  }
  // EBLP
  else if (memcmp(buffer, "\xFE" "\xED" "\xBE" "\xEF", 4) == 0) {
    proto = "EBLP";
    height = ntohs(*(uint16_t *) (buffer + 10));
    width = ntohs(*(uint16_t *) (buffer + 8));
    channels = 1;
    maxval = 255;
  }
  // MCUF
  else if (memcmp(buffer, "\x23" "\x54" "\x26" "\x66", 4) == 0) {
    proto = "MCUF";
    height = ntohs(*(uint16_t *) (buffer + 4));
    width = ntohs(*(uint16_t *) (buffer + 6));
    channels = ntohs(*(uint16_t *) (buffer + 8));
    maxval = ntohs(*(uint16_t *) (buffer + 10));
  }
  // unknown
  else {
    proto = "unknown";
    height = 0;
    width = 0;
    channels = 0;
    maxval = 1;
  }

  // too short
  if (maxval < 1 || (unsigned int)len < 12 + height * width * channels) {
    proto = "invalid";
    height = 0;
    width = 0;
    channels = 0;
    maxval = 1;
  }
  // show reception
  t = time(NULL);
  localtime_r(&t, &tm);
  snprintf(text, sizeof(text), "%s %ux%u-%u/%u from %s:%hu at %02u:%02u:%02u",
           proto, width, height, channels, maxval + 1,
           inet_ntoa(addr.sin_addr), ntohs(addr.sin_port),
           tm.tm_hour, tm.tm_min, tm.tm_sec);
  net_show_reception(text);

  // save frame
  net_frame_height = height;
  net_frame_width = width;
  net_frame_channels = channels;
  net_frame_maxval = maxval;
  memcpy(net_frame_data, buffer + 12,
         (unsigned int)len - 12 < sizeof(net_frame_data) ?
           (unsigned int)len - 12 : sizeof(net_frame_data));

  // show new frame
  net_redraw();

  return TRUE;

  (void)source;
  (void)condition;
  (void)data;
}

// shutdown socket, ...
static void net_socket_close()
{
  if (net_sock != -1) {
    g_source_remove(net_io_watch);
    g_io_channel_unref(net_io_channel);
    close(net_sock);
    net_sock = -1;
  }
}

// create new socket, ...
static void net_socket_new()
{
  net_socket_close();

  net_sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
  if (net_sock == -1)
    return;
  if (fcntl(net_sock, F_SETFL, O_NONBLOCK) != 0) {
    close(net_sock);
    net_sock = -1;
    return;
  }
  if (bind(net_sock, (struct sockaddr *)&net_listen, sizeof(net_listen)) ==
      -1) {
    close(net_sock);
    net_sock = -1;
    return;
  }

  net_io_channel = g_io_channel_unix_new(net_sock);
  if (net_io_channel == NULL) {
    close(net_sock);
    net_sock = -1;
    return;
  }
  g_io_channel_set_encoding(net_io_channel, NULL, NULL);
  g_io_channel_set_buffered(net_io_channel, FALSE);
  net_io_watch =
      g_io_add_watch(net_io_channel, G_IO_IN | G_IO_ERR, net_packet, NULL);
}

// initialize
void net_init()
{
  // initialize listen address
  net_listen.sin_family = AF_INET;
  net_listen.sin_port = htons(2323);
  net_listen.sin_addr.s_addr = INADDR_ANY;

  // initialize send address
  net_listen.sin_family = AF_INET;
  net_listen.sin_port = htons(0);
  net_listen.sin_addr.s_addr = INADDR_NONE;

  // initialize protocol
  net_protocol = 2;

  // initialize zoom
  net_zoom = 16;

  // no socket yet
  net_sock = -1;
}

// cleanup
void net_exit()
{
  net_socket_close();
}

// get listen address
void net_get_listen(char *pBuffer, unsigned int bufferLen)
{
  if (net_listen.sin_addr.s_addr != INADDR_NONE)
    snprintf(pBuffer, bufferLen, "%s:%hu",
             inet_ntoa(net_listen.sin_addr), ntohs(net_listen.sin_port));
  else
    pBuffer[0] = 0;
}

// set new listen address
void net_set_listen(char *pAddr)
{
  char *pColon;
  unsigned short port;

  net_send_close();     // send a close

  net_listen.sin_addr.s_addr = INADDR_NONE;

  if ((pColon = strchr(pAddr, ':')) == NULL)
    return;
  *pColon = 0;
  if (sscanf(pColon + 1, "%hu", &port) != 1)
    return;
  net_listen.sin_port = htons(port);
  net_listen.sin_addr.s_addr = inet_addr(pAddr);

  net_socket_new();
  net_show_reception("");       // remove reception text
  net_frame_height = 0; // remove frame
  net_frame_width = 0;
  net_frame_channels = 0;
  net_frame_maxval = 0;
  net_redraw();

  net_send_request();   // send a request
}

// get send address
void net_get_send(char *pBuffer, unsigned int bufferLen)
{
  if (net_send.sin_addr.s_addr != INADDR_NONE)
    snprintf(pBuffer, bufferLen, "%s:%hu",
             inet_ntoa(net_send.sin_addr), ntohs(net_send.sin_port));
  else
    pBuffer[0] = 0;
}

// set new send address
void net_set_send(char *pAddr)
{
  char *pColon;
  unsigned short port;

  net_send_close();     // send a close

  net_send.sin_addr.s_addr = INADDR_NONE;

  if ((pColon = strchr(pAddr, ':')) == NULL)
    return;
  *pColon = 0;
  if (sscanf(pColon + 1, "%hu", &port) != 1)
    return;
  net_send.sin_port = htons(port);
  net_send.sin_addr.s_addr = inet_addr(pAddr);

  net_show_reception("");       // remove reception text
  net_frame_height = 0; // remove frame
  net_frame_width = 0;
  net_frame_channels = 0;
  net_frame_maxval = 0;
  net_redraw();

  net_send_request();   // send a request
}

// get protocol
unsigned int net_get_protocol()
{
  return net_protocol;
}

// set new protocol
void net_set_protocol(unsigned int protocol)
{
  net_send_close();     // send a close

  if (protocol < net_protocol_cnt)
    net_protocol = protocol;

  net_show_reception("");       // remove reception text
  net_frame_height = 0; // remove frame
  net_frame_width = 0;
  net_frame_channels = 0;
  net_frame_maxval = 0;
  net_redraw();

  net_send_request();   // send a request
}

// get zoom factor
unsigned int net_get_zoom()
{
  return net_zoom;
}

// set new zoom factor
void net_set_zoom(unsigned int zoom)
{
  if (zoom < 1)
    zoom = 1;
  if (zoom > 30)
    zoom = 30;
  net_zoom = zoom;
  net_redraw();
}
